<?php
/**
 * Admin Login Handler
 *
 * This script processes the admin login form submission.
 * It is separate from the user login to ensure a clear distinction
 * between user and administrator roles.
 */

// Start the session to manage admin login state.
session_start();

// Include core files
require_once '../config.php';
require_once '../app/helpers.php';

// 1. VERIFY REQUEST METHOD
// -----------------------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ../admin/index.php?page=login');
    exit();
}

// 2. RETRIEVE & SANITIZE FORM DATA
// -----------------------------------------------------------------------------
$email = $_POST['email'] ?? '';
$password = $_POST['password'] ?? '';

// 3. VALIDATE INPUT
// -----------------------------------------------------------------------------
if (empty($email) || empty($password)) {
    $_SESSION['error_message'] = "Email and password are required.";
    header('Location: ../admin/index.php?page=login');
    exit();
}

// 4. DATABASE AUTHENTICATION
// -----------------------------------------------------------------------------
try {
    $pdo = get_db_connection();

    // Prepare a statement to find the admin by their email address from the `admins` table.
    $stmt = $pdo->prepare("SELECT id, full_name, email, password FROM admins WHERE email = ?");
    $stmt->execute([$email]);
    $admin = $stmt->fetch();

    // Check if an admin was found AND if the provided password matches the hashed password.
    if ($admin && password_verify($password, $admin['password'])) {

        // --- SUCCESSFUL ADMIN LOGIN ---

        // Regenerate session ID to prevent session fixation attacks.
        session_regenerate_id(true);

        // Store admin information in the session.
        // It's crucial to use a different session key (e.g., 'admin_id') to
        // distinguish from a regular user session.
        $_SESSION['admin_id'] = $admin['id'];
        $_SESSION['admin_name'] = $admin['full_name'];
        $_SESSION['is_admin_logged_in'] = true;

        // Redirect to the admin dashboard.
        header('Location: ../admin/index.php?page=dashboard');
        exit();

    } else {
        // --- FAILED ADMIN LOGIN ---
        // Show a generic error message to prevent user enumeration.
        $_SESSION['error_message'] = "Invalid email or password.";
        header('Location: ../admin/index.php?page=login');
        exit();
    }

} catch (PDOException $e) {
    // For a production environment, log this error instead of displaying it.
    // error_log("Admin login failed: " . $e->getMessage());
    $_SESSION['error_message'] = "An unexpected error occurred. Please try again later.";
    header('Location: ../admin/index.php?page=login');
    exit();
}
