<?php
/**
 * Forgot Password Handler
 *
 * This script processes the forgot password form submission.
 * It validates the email, generates a secure reset token,
 * stores it in the database, and sends a password reset email.
 */

// Start the session to manage user feedback (messages).
session_start();

// Include core files
require_once '../config.php';
require_once '../app/helpers.php'; // Contains send_email() and get_db_connection()

// 1. VERIFY REQUEST METHOD
// -----------------------------------------------------------------------------
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ../index.php?page=forgot_password');
    exit();
}

// 2. RETRIEVE & VALIDATE EMAIL
// -----------------------------------------------------------------------------
$email = trim($_POST['email'] ?? '');

if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    $_SESSION['error_message'] = "Please enter a valid email address.";
    header('Location: ../index.php?page=forgot_password');
    exit();
}

// 3. DATABASE & TOKEN GENERATION
// -----------------------------------------------------------------------------
try {
    $pdo = get_db_connection();

    // Find the user by email.
    $stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? AND email_verified = 1");
    $stmt->execute([$email]);
    $user = $stmt->fetch();

    // IMPORTANT SECURITY NOTE:
    // We will show a success message even if the user is NOT found.
    // This prevents "user enumeration", where an attacker could otherwise
    // figure out which email addresses are registered in our system.
    if ($user) {
        // --- USER FOUND: Generate and send the reset link ---

        // Generate a cryptographically secure token.
        $reset_token = bin2hex(random_bytes(32));

        // Set an expiration time for the token (e.g., 1 hour from now).
        $expires = new DateTime('now', new DateTimeZone('Asia/Jakarta'));
        $expires->add(new DateInterval('PT1H')); // P-Period, T-Time, 1-One, H-Hour
        $expires_formatted = $expires->format('Y-m-d H:i:s');

        // Store the hashed token and its expiration date in the database.
        // We hash the token so that if the database is compromised,
        // the tokens cannot be easily used.
        $hashed_token = hash('sha256', $reset_token);
        $update_stmt = $pdo->prepare(
            "UPDATE users SET reset_token = ?, reset_token_expires = ? WHERE id = ?"
        );
        $update_stmt->execute([$hashed_token, $expires_formatted, $user['id']]);

        // 4. SEND RESET EMAIL
        // -----------------------------------------------------------------------------
        // Note: The link contains the *unhashed* token. We will hash it again
        // when the user clicks the link to verify it.
        $reset_link = SITE_URL . "/reset_password.php?token=" . $reset_token;
        $email_subject = "Password Reset Request for " . SITE_NAME;
        $email_body = "
            <h2>Password Reset Request</h2>
            <p>We received a request to reset your password. If you did not make this request, you can safely ignore this email.</p>
            <p>To reset your password, please click the link below. This link is valid for 1 hour:</p>
            <p><a href='{$reset_link}'>{$reset_link}</a></p>
        ";

        send_email($email, $email_subject, $email_body);
    }

    // 5. REDIRECT WITH SUCCESS MESSAGE
    // -----------------------------------------------------------------------------
    // This message is shown whether the user was found or not.
    $_SESSION['success_message'] = "If an account with that email exists, a password reset link has been sent.";
    header('Location: ../index.php?page=login');
    exit();

} catch (PDOException $e) {
    // In production, log the error.
    // error_log("Forgot password failed: " . $e->getMessage());
    $_SESSION['error_message'] = "An unexpected error occurred. Please try again later.";
    header('Location: ../index.php?page=forgot_password');
    exit();
}
