<?php
/**
 * Get AI Learning Job Status API
 *
 * This script is called via AJAX from the admin learning page to get the
 * real-time status of the most recent background job for a specific type (CV or Job Poster).
 * It returns a JSON object with the job's progress and status.
 */

// Set the content type to JSON for the response.
header('Content-Type: application/json');

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../config.php';
require_once '../app/helpers.php';

// 1. SECURITY CHECK: Ensure only logged-in admins can access this API endpoint.
if (!is_admin()) {
    // Return a JSON error response if not authorized.
    echo json_encode(['status' => 'error', 'message' => 'Unauthorized access.']);
    exit();
}

// 2. GET & VALIDATE INPUT
$job_type = $_GET['type'] ?? '';
if ($job_type !== 'cv' && $job_type !== 'job_poster') {
    echo json_encode(['status' => 'error', 'message' => 'Invalid job type specified.']);
    exit();
}

// 3. FETCH LATEST JOB STATUS FROM DATABASE
try {
    $pdo = get_db_connection();

    // Find the most recent job for the specified type.
    $stmt = $pdo->prepare("
        SELECT * FROM ai_learning_jobs 
        WHERE job_type = ? 
        ORDER BY id DESC 
        LIMIT 1
    ");
    $stmt->execute([$job_type]);
    $job = $stmt->fetch();

    $response = [];

    if ($job) {
        // --- JOB FOUND ---
        $response['status'] = 'success';
        $response['job_status'] = $job['status'];
        $response['message'] = $job['message'];
        $response['processed_items'] = (int)$job['processed_items'];
        $response['total_items'] = (int)$job['total_items'];
        
        // Calculate percentage progress
        if ($job['total_items'] > 0 && ($job['status'] === 'processing' || $job['status'] === 'completed')) {
            $response['progress_percent'] = round(($job['processed_items'] / $job['total_items']) * 100);
        } else {
            $response['progress_percent'] = 0;
        }

        // Format dates for display
        $response['created_at'] = $job['created_at'] ? date('d M Y, H:i', strtotime($job['created_at'])) : null;
        $response['completed_at'] = $job['completed_at'] ? date('d M Y, H:i', strtotime($job['completed_at'])) : null;

    } else {
        // --- NO JOB FOUND ---
        // This is a normal state if no job has been run for this type yet.
        $response['status'] = 'success';
        $response['job_status'] = 'not_found';
        $response['message'] = 'No learning process has been initiated for this category yet.';
    }

    // 4. SEND JSON RESPONSE
    echo json_encode($response);
    exit();

} catch (Exception $e) {
    // Return a JSON error if the database query fails.
    http_response_code(500); // Internal Server Error
    echo json_encode(['status' => 'error', 'message' => 'Database error: ' . $e->getMessage()]);
    exit();
}

