<?php
/**
 * AI Interview Results Handler
 *
 * This script receives the interview data (questions and answer transcripts)
 * from the front-end via an AJAX POST request. It saves this data to the
 * database and updates the user's progress to the final step.
 */

header('Content-Type: application/json');

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../config.php';
require_once '../app/helpers.php';

// 1. SECURITY CHECKS
// -----------------------------------------------------------------------------
if (!is_logged_in() || $_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['status' => 'error', 'message' => 'Unauthorized access.']);
    exit();
}

// 2. RECEIVE AND DECODE JSON DATA
// -----------------------------------------------------------------------------
$input = json_decode(file_get_contents('php://input'), true);
$interviewData = $input['interviewData'] ?? null;

if (!$interviewData || !is_array($interviewData)) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid or missing interview data.']);
    exit();
}

// 3. SAVE INTERVIEW DATA TO DATABASE
// -----------------------------------------------------------------------------
try {
    $pdo = get_db_connection();
    $user_id = $_SESSION['user_id'];

    $pdo->beginTransaction();

    // Prepare a statement to insert each question/answer pair.
    $stmt_insert = $pdo->prepare(
        "INSERT INTO interview_results (user_id, question_text, answer_transcript) VALUES (?, ?, ?)"
    );

    foreach ($interviewData as $entry) {
        $question = $entry['question'] ?? '';
        $answer = $entry['answer'] ?? '';

        // Insert each Q&A pair into the database.
        if (!empty($question) || !empty($answer)) { // Avoid saving empty records
            $stmt_insert->execute([$user_id, $question, $answer]);
        }
    }

    // Advance the user to the final step (viewing results).
    $stmt_step = $pdo->prepare("UPDATE users SET current_step = 5 WHERE id = ?");
    $stmt_step->execute([$user_id]);

    $pdo->commit();

    // 4. SEND SUCCESS RESPONSE
    // -----------------------------------------------------------------------------
    echo json_encode(['status' => 'success', 'message' => 'Interview results saved successfully.']);
    exit();

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    // Return a JSON error for the front-end to handle.
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Database error: ' . $e->getMessage()]);
    exit();
}

