<?php
/**
 * Psychological Test Handler
 *
 * This script processes the user's submitted answers from the psychological test.
 * It calculates the total score based on the selected answers and updates the
 * user's progress to the next step (AI Interview).
 */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../config.php';
require_once '../app/helpers.php';

// 1. SECURITY CHECKS
// -----------------------------------------------------------------------------
if (!is_logged_in() || $_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['error_message'] = "Unauthorized access.";
    header('Location: ../index.php?page=login');
    exit();
}

// 2. RETRIEVE & VALIDATE FORM DATA
// -----------------------------------------------------------------------------
$test_id = filter_input(INPUT_POST, 'test_id', FILTER_VALIDATE_INT);
$submitted_answers = $_POST['answers'] ?? [];

if (!$test_id || empty($submitted_answers)) {
    // Redirect back to the test page if data is missing.
    // A more specific error could be set in the session if needed.
    header('Location: ../index.php?page=psy_test');
    exit();
}

// 3. CALCULATE SCORE
// -----------------------------------------------------------------------------
try {
    $pdo = get_db_connection();
    $total_score = 0;

    // Get the scores for all the answers the user selected.
    // We create a string of placeholders for the IN clause to prevent SQL injection.
    $placeholders = implode(',', array_fill(0, count($submitted_answers), '?'));
    
    $stmt_scores = $pdo->prepare("SELECT score FROM test_answers WHERE id IN ($placeholders)");
    $stmt_scores->execute(array_values($submitted_answers));
    
    $scores = $stmt_scores->fetchAll(PDO::FETCH_COLUMN);

    // Sum up all the fetched scores.
    $total_score = array_sum($scores);

    // 4. SAVE RESULTS AND UPDATE USER PROGRESS
    // -----------------------------------------------------------------------------
    $pdo->beginTransaction();

    // Save the final test result.
    $stmt_result = $pdo->prepare("INSERT INTO user_test_results (user_id, test_id, total_score) VALUES (?, ?, ?)");
    $stmt_result->execute([$_SESSION['user_id'], $test_id, $total_score]);

    // Advance the user to the next step (AI Interview).
    $stmt_step = $pdo->prepare("UPDATE users SET current_step = 4 WHERE id = ?");
    $stmt_step->execute([$_SESSION['user_id']]);

    $pdo->commit();

    // 5. REDIRECT TO THE NEXT STEP
    // -----------------------------------------------------------------------------
    header('Location: ../index.php?page=interview');
    exit();

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    // Set an error message to be displayed on the test page.
    $_SESSION['error_message'] = "An error occurred while submitting your test: " . $e->getMessage();
    header('Location: ../index.php?page=psy_test');
    exit();
}

