<?php

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

if (session_status() === PHP_SESSION_NONE) {

/**
 * Handler: Test Management (Add, Edit, Delete, Questions)
 *
 * PERBAIKAN:
 * - Aksi 'add_test' dan 'edit_test' sekarang dibungkus dalam
 * transaksi (beginTransaction/commit) untuk MEMAKSA
 * database menyimpan perubahan. Ini memperbaiki bug "gagal senyap".
 * - Memastikan 'admin_id' diambil dengan benar dari sesi.
 * - Memulihkan logika 'add_question' dan 'delete_question' dari file asli.
 */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../config.php';
require_once '../app/helpers.php';

// Keamanan: Pastikan hanya admin yang dapat mengakses ini
if (!is_admin()) {
    $_SESSION['error_message'] = "You are not authorized.";
    header('Location: ../admin/index.php?page=login');
    exit();
}

// Ambil ID admin dari sesi.
$admin_id = $_SESSION['admin_id'] ?? 0;
if ($admin_id == 0) {
    // Ini adalah Foreign Key, jadi jika 0, INSERT akan GAGAL.
    $_SESSION['error_message'] = "Admin session invalid. Please log in again.";
    header('Location: ../admin/index.php?page=login');
    exit();
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ../admin/index.php?page=dashboard');
    exit();
}

$action = $_POST['action'] ?? '';
$pdo = get_db_connection();

try {
    switch ($action) {
        // --- Aksi: Menambahkan Tes Baru (FIXED) ---
        case 'add_test':
            $test_name = trim($_POST['test_name'] ?? '');
            $description = trim($_POST['description'] ?? '');
            $test_type = trim($_POST['test_type'] ?? 'simple');
            
            if (empty($test_name)) {
                throw new Exception("Test name is required.");
            }

            // PERBAIKAN: Memulai Transaksi
            $pdo->beginTransaction();

            $sql = "INSERT INTO psychological_tests 
                        (test_name, name, description, test_type, module_version, created_by_admin_id) 
                    VALUES 
                        (?, ?, ?, ?, ?, ?)";
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute([
                $test_name,
                '', // Kolom 'name' (NOT NULL)
                $description,
                $test_type,
                '', // Kolom 'module_version' (NOT NULL)
                $admin_id
            ]);
            
            // PERBAIKAN: Menyelesaikan Transaksi
            $pdo->commit();

            $_SESSION['success_message'] = "New test '{$test_name}' has been successfully created.";
            header('Location: ../admin/index.php?page=tests');
            exit();

        // --- Aksi: Mengedit Tes yang Ada (FIXED) ---
        case 'edit_test':
            $test_id = filter_input(INPUT_POST, 'test_id', FILTER_VALIDATE_INT);
            $test_name = trim($_POST['test_name'] ?? '');
            $description = trim($_POST['description'] ?? '');
            $test_type = trim($_POST['test_type'] ?? 'simple');
            
            if (!$test_id || empty($test_name)) {
                throw new Exception("Invalid data provided for editing the test.");
            }
            
            // PERBAIKAN: Memulai Transaksi
            $pdo->beginTransaction();
            
            $sql = "UPDATE psychological_tests SET 
                        test_name = ?, 
                        description = ?, 
                        test_type = ? 
                    WHERE id = ?";
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$test_name, $description, $test_type, $test_id]);
            
            // PERBAIKAN: Menyelesaikan Transaksi
            $pdo->commit();
            
            $_SESSION['success_message'] = "Test details have been successfully updated.";
            header('Location: ../admin/index.php?page=edit_test&id=' . $test_id);
            exit();

        // --- Aksi: Menghapus Tes ---
        case 'delete_test':
            $test_id = filter_input(INPUT_POST, 'test_id', FILTER_VALIDATE_INT);
            if (!$test_id) {
                throw new Exception("Invalid test ID for deletion.");
            }
            
            $pdo->beginTransaction();
            $stmt = $pdo->prepare("DELETE FROM psychological_tests WHERE id = ?");
            $stmt->execute([$test_id]);
            $pdo->commit();
            
            $_SESSION['success_message'] = "The test has been deleted.";
            header('Location: ../admin/index.php?page=tests');
            exit();

        // --- Aksi: Menambahkan Pertanyaan (Dipulihkan) ---
        case 'add_question':
            $test_id = filter_input(INPUT_POST, 'test_id', FILTER_VALIDATE_INT);
            $question_text = trim($_POST['question_text'] ?? '');
            $answers = $_POST['answers'] ?? [];
            $scores = $_POST['scores'] ?? [];

            if (!$test_id || empty($question_text) || count($answers) < 2) {
                throw new Exception("A question and at least two answers are required.");
            }

            $pdo->beginTransaction();
            $stmt_q = $pdo->prepare("INSERT INTO test_questions (test_id, question_text) VALUES (?, ?)");
            $stmt_q->execute([$test_id, $question_text]);
            $question_id = $pdo->lastInsertId();

            $stmt_a = $pdo->prepare("INSERT INTO test_answers (question_id, answer_text, score) VALUES (?, ?, ?)");
            foreach ($answers as $index => $answer_text) {
                $answer_text = trim($answer_text);
                $score = filter_var($scores[$index] ?? 0, FILTER_VALIDATE_INT);
                if (!empty($answer_text)) {
                    $stmt_a->execute([$question_id, $answer_text, $score]);
                }
            }
            
            $stmt_count = $pdo->prepare("UPDATE psychological_tests SET question_count = (SELECT COUNT(*) FROM test_questions WHERE test_id = ?) WHERE id = ?");
            $stmt_count->execute([$test_id, $test_id]);

            $pdo->commit();
            $_SESSION['success_message'] = "New question has been successfully added.";
            header('Location: ../admin/index.php?page=manage_questions&test_id=' . $test_id);
            exit();

        // --- Aksi: Menghapus Pertanyaan (Dipulihkan) ---
        case 'delete_question':
            $question_id = filter_input(INPUT_POST, 'question_id', FILTER_VALIDATE_INT);
            $test_id = filter_input(INPUT_POST, 'test_id', FILTER_VALIDATE_INT); 
            if (!$question_id || !$test_id) {
                throw new Exception("Invalid question ID for deletion.");
            }
            
            $pdo->beginTransaction();
            $stmt = $pdo->prepare("DELETE FROM test_questions WHERE id = ?");
            $stmt->execute([$question_id]);
            $stmt_count = $pdo->prepare("UPDATE psychological_tests SET question_count = (SELECT COUNT(*) FROM test_questions WHERE test_id = ?) WHERE id = ?");
            $stmt_count->execute([$test_id, $test_id]);
            $pdo->commit();
            
            $_SESSION['success_message'] = "The question has been successfully deleted.";
            header('Location: ../admin/index.php?page=manage_questions&test_id=' . $test_id);
            exit();

        default:
            throw new Exception("Invalid action specified.");
    }

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $_SESSION['error_message'] = "An error occurred: " . $e->getMessage();
    
    $redirect_page = 'tests'; // default
    if ($action === 'add_test') {
        $redirect_page = 'add_test';
    } elseif ($action === 'edit_test' && isset($_POST['test_id'])) {
        $redirect_page = 'edit_test&id=' . $_POST['test_id'];
    } elseif (isset($_POST['test_id'])) {
        $redirect_page = 'manage_questions&test_id=' . $_POST['test_id'];
    }
    
    header('Location: ../admin/index.php?page=' . $redirect_page);
    exit();
}