<?php
/**
 * AI Learning Handler (Updated with Recursive Folder Scanning)
 *
 * This script connects to Google Drive, finds new documents, processes them,
 * and stores their "learnings" (embeddings) in the database.
 * It now recursively scans through all subfolders to learn from every document.
 */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../config.php';
require_once '../vendor/autoload.php';
require_once '../app/helpers.php';

if (!is_admin()) {
    $_SESSION['error_message'] = "You do not have permission to access that page.";
    header('Location: ../admin/index.php?page=login');
    exit();
}

$folder_type = $_POST['folder_type'] ?? '';
if ($folder_type !== 'cv' && $folder_type !== 'job_poster') {
    $_SESSION['error_message'] = "Invalid learning source specified.";
    header('Location: ../admin/index.php?page=learning');
    exit();
}

$initial_folder_id = ($folder_type === 'cv') ? GOOGLE_DRIVE_CV_FOLDER_ID : GOOGLE_DRIVE_JOB_POSTER_FOLDER_ID;
$folder_name_for_message = ($folder_type === 'cv') ? "CVs" : "Job Posters";

try {
    $client = new Google_Client();
    $client->setAuthConfig('../credentials.json');
    $client->setRedirectUri(GOOGLE_REDIRECT_URI);
    $client->setAccessType('offline');
    $client->setPrompt('select_account consent');
    $client->addScope(Google_Service_Drive::DRIVE_READONLY);

    $tokenPath = '../token.json';
    if (file_exists($tokenPath)) {
        $client->setAccessToken(json_decode(file_get_contents($tokenPath), true));
    }

    if ($client->isAccessTokenExpired()) {
        if ($client->getRefreshToken()) {
            $client->fetchAccessTokenWithRefreshToken($client->getRefreshToken());
            file_put_contents($tokenPath, json_encode($client->getAccessToken()));
        } else {
            header('Location: ' . $client->createAuthUrl());
            exit();
        }
    }

    $drive_service = new Google_Service_Drive($client);
    $pdo = get_db_connection();

    // --- RECURSIVE FUNCTION DEFINITION ---
    // This function will scan a folder, process its files, and dive into subfolders.
    function processFilesInFolder($folder_id, $drive_service, $pdo, &$newly_processed_count, &$total_files_scanned) {
        $check_stmt = $pdo->prepare("SELECT 1 FROM processed_files WHERE file_id = ?");

        $exportable_mime_types = [
            'application/vnd.google-apps.document',
            'application/vnd.google-apps.spreadsheet',
            'application/vnd.google-apps.presentation'
        ];
        
        $pageToken = null;
        do {
            $response = $drive_service->files->listFiles([
                'q' => "'{$folder_id}' in parents and trashed=false",
                'fields' => 'nextPageToken, files(id, name, mimeType)',
                'pageToken' => $pageToken
            ]);

            foreach ($response->getFiles() as $item) {
                $total_files_scanned++;
                $file_id = $item->getId();
                $file_name = $item->getName();
                $mime_type = $item->getMimeType();

                // If the item is a folder, recurse into it.
                if ($mime_type === 'application/vnd.google-apps.folder') {
                    processFilesInFolder($file_id, $drive_service, $pdo, $newly_processed_count, $total_files_scanned);
                    continue; // Move to the next item in the current folder
                }
                
                // If it's a file, check if it's already processed.
                $check_stmt->execute([$file_id]);
                if ($check_stmt->fetch()) {
                    continue; // Already processed.
                }

                $newly_processed_count++;
                
                // --- SMART DOWNLOAD/EXPORT LOGIC ---
                if (in_array($mime_type, $exportable_mime_types)) {
                    $content = $drive_service->files->export($file_id, 'text/plain', ['alt' => 'media']);
                } else {
                    $content = $drive_service->files->get($file_id, ['alt' => 'media']);
                }
                
                $file_content = $content->getBody()->getContents();
                $text_content = $file_content;
                $embedding_vector = "SIMULATED_VECTOR_FOR_" . substr(md5($text_content), 0, 10);

                $pdo->beginTransaction();
                $processed_stmt = $pdo->prepare("INSERT INTO processed_files (file_id, file_name) VALUES (?, ?)");
                $processed_stmt->execute([$file_id, $file_name]);
                $embedding_stmt = $pdo->prepare("INSERT INTO vector_embeddings (file_id, content_chunk, embedding) VALUES (?, ?, ?)");
                $embedding_stmt->execute([$file_id, mb_substr($text_content, 0, 500), $embedding_vector]);
                $pdo->commit();
            }
            $pageToken = $response->getNextPageToken();
        } while ($pageToken != null);
    }

    // --- MAIN EXECUTION ---
    $newly_processed_count = 0;
    $total_files_scanned = 0;
    // Start the recursive process from the initial folder
    processFilesInFolder($initial_folder_id, $drive_service, $pdo, $newly_processed_count, $total_files_scanned);

    // --- REDIRECT WITH INFORMATIVE MESSAGE ---
    $message = "Learning update complete for {$folder_name_for_message}. ";
    $message .= "Scanned a total of {$total_files_scanned} items (files and folders) and learned from {$newly_processed_count} new files.";
    
    $_SESSION['success_message'] = $message;
    header('Location: ../admin/index.php?page=learning');
    exit();

} catch (Exception $e) {
    $_SESSION['error_message'] = "An error occurred during the learning process: " . $e->getMessage();
    header('Location: ../admin/index.php?page=learning');
    exit();
}

