<?php
/**
 * AI Learning Job Creation Handler
 *
 * This script is executed when an admin clicks the "Update Learnings" button.
 * Instead of doing the heavy lifting itself, its only job is to create a new
 * 'pending' job in the `ai_learning_jobs` table in the database.
 * The background cron job will then pick up this job and process it.
 */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../config.php';
require_once '../app/helpers.php';

// 1. SECURITY CHECK: Ensure only logged-in admins can run this.
if (!is_admin()) {
    $_SESSION['error_message'] = "You do not have permission to perform this action.";
    header('Location: ../admin/index.php?page=login');
    exit();
}

// 2. VALIDATE INPUT
$folder_type = $_POST['folder_type'] ?? '';
if ($folder_type !== 'cv' && $folder_type !== 'job_poster') {
    $_SESSION['error_message'] = "Invalid learning source specified.";
    header('Location: ../admin/index.php?page=learning');
    exit();
}

try {
    $pdo = get_db_connection();

    // 3. CHECK FOR EXISTING PENDING/PROCESSING JOBS for the same type
    // This prevents creating duplicate jobs if one is already running or waiting.
    $stmt_check = $pdo->prepare("SELECT id FROM ai_learning_jobs WHERE job_type = ? AND (status = 'pending' OR status = 'processing')");
    $stmt_check->execute([$folder_type]);
    
    if ($stmt_check->fetch()) {
        $_SESSION['info_message'] = "A learning process for this category is already in progress or waiting to be processed.";
        header('Location: ../admin/index.php?page=learning');
        exit();
    }

    // 4. CREATE THE NEW JOB
    // Insert a new row into the jobs table with a 'pending' status.
    $stmt_insert = $pdo->prepare("INSERT INTO ai_learning_jobs (job_type, status) VALUES (?, 'pending')");
    $stmt_insert->execute([$folder_type]);
    
    $job_id = $pdo->lastInsertId();

    // 5. REDIRECT WITH SUCCESS MESSAGE
    // The user is notified that the process has started and will run in the background.
    if($job_id) {
        $_SESSION['success_message'] = "The learning process has been initiated. It will run in the background. You can monitor the progress on this page.";
    } else {
        throw new Exception("Failed to create a new learning job in the database.");
    }
    
    header('Location: ../admin/index.php?page=learning');
    exit();

} catch (Exception $e) {
    $_SESSION['error_message'] = "An error occurred while initiating the learning process: " . $e->getMessage();
    header('Location: ../admin/index.php?page=learning');
    exit();
}

