<?php
/**
 * User Upload Handler
 *
 * This script processes the file uploads from the user's Step 1.
 * It validates file types (including images), size, saves them securely,
 * and updates the user's progress in the database to the payment step.
 */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../config.php';
require_once '../app/helpers.php';

// 1. SECURITY CHECKS
// -----------------------------------------------------------------------------
if (!is_logged_in()) {
    $_SESSION['upload_error'] = "You must be logged in to upload files.";
    header('Location: ../index.php?page=upload');
    exit();
}
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ../index.php?page=upload');
    exit();
}

// 2. FILE PROCESSING FUNCTION
// -----------------------------------------------------------------------------
function process_uploaded_file($file_key, $user_id, $file_type) {
    $allowed_types = ['application/pdf', 'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document', 'text/plain', 'image/jpeg', 'image/png'];
    $allowed_extensions = ['pdf', 'doc', 'docx', 'txt', 'jpg', 'jpeg', 'png'];
    $max_size = 2 * 1024 * 1024; // 2 MB

    if (!isset($_FILES[$file_key]) || $_FILES[$file_key]['error'] !== UPLOAD_ERR_OK) {
        throw new Exception("Error uploading the {$file_type} file. Please try again.");
    }

    $file = $_FILES[$file_key];
    $file_extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

    if ($file['size'] > $max_size) {
        throw new Exception("The {$file_type} file exceeds the 2MB size limit.");
    }
    if (!in_array($file_extension, $allowed_extensions)) {
        throw new Exception("Invalid file type for {$file_type}. Only PDF, DOC, DOCX, TXT, JPG, PNG are allowed.");
    }

    $upload_dir = __DIR__ . '/../public/uploads/user_' . $user_id . '/';
    if (!is_dir($upload_dir)) {
        if (!mkdir($upload_dir, 0755, true)) {
            throw new Exception('Failed to create user upload directory. Check permissions.');
        }
    }

    $new_filename = uniqid($file_type . '_', true) . '.' . $file_extension;
    $stored_filepath = $upload_dir . $new_filename;

    if (!move_uploaded_file($file['tmp_name'], $stored_filepath)) {
        throw new Exception("Failed to save the uploaded {$file_type} file.");
    }
    
    return [
        'original_filename' => basename($file['name']),
        'stored_filepath' => 'public/uploads/user_' . $user_id . '/' . $new_filename
    ];
}

// 3. MAIN EXECUTION
// -----------------------------------------------------------------------------
try {
    $pdo = get_db_connection();
    $user_id = $_SESSION['user_id'];

    $cv_details = process_uploaded_file('cv_file', $user_id, 'CV');
    $job_ad_details = process_uploaded_file('job_ad_file', $user_id, 'Job Ad');

    $pdo->beginTransaction();

    // Clear previous uploads for this user to ensure only the latest are stored for the session
    $stmt_clear = $pdo->prepare("DELETE FROM user_uploads WHERE user_id = ?");
    $stmt_clear->execute([$user_id]);

    $stmt_cv = $pdo->prepare("INSERT INTO user_uploads (user_id, file_type, original_filename, stored_filepath) VALUES (?, 'cv', ?, ?)");
    $stmt_cv->execute([$user_id, $cv_details['original_filename'], $cv_details['stored_filepath']]);

    $stmt_job = $pdo->prepare("INSERT INTO user_uploads (user_id, file_type, original_filename, stored_filepath) VALUES (?, 'job_ad', ?, ?)");
    $stmt_job->execute([$user_id, $job_ad_details['original_filename'], $job_ad_details['stored_filepath']]);

    // --- PAYMENT BYPASS REMOVED ---
    // The user is now correctly advanced to step 2 (payment).
    $next_step = 3; 
    $redirect_page = 'psy_test';

    $stmt_step = $pdo->prepare("UPDATE users SET current_step = ? WHERE id = ?");
    $stmt_step->execute([$next_step, $user_id]);

    $pdo->commit();

    // Redirect to the actual payment page
    header('Location: ../index.php?page=' . $redirect_page);
    exit();

} catch (Exception $e) {
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    $_SESSION['upload_error'] = $e->getMessage();
    header('Location: ../index.php?page=upload');
    exit();
}

