<?php
/**
 * User Login Handler
 *
 * This script processes the user login form submission.
 * It validates the input, checks credentials against the database,
 * and manages the user's session.
 */

// Start the session to manage user login state.
session_start();

// Include core files
require_once '../config.php';
require_once '../app/helpers.php';

// 1. VERIFY REQUEST METHOD
// -----------------------------------------------------------------------------
// We only process POST requests. If it's a GET or other method,
// redirect back to the main page.
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ../index.php');
    exit();
}

// 2. RETRIEVE & SANITIZE FORM DATA
// -----------------------------------------------------------------------------
$email = $_POST['email'] ?? '';
$password = $_POST['password'] ?? '';
$remember_me = isset($_POST['remember_me']);

// 3. VALIDATE INPUT
// -----------------------------------------------------------------------------
// Simple validation to ensure fields are not empty.
if (empty($email) || empty($password)) {
    // Store an error message in the session to display on the login page.
    $_SESSION['error_message'] = "Email and password are required.";
    header('Location: ../index.php?page=login');
    exit();
}

// 4. DATABASE AUTHENTICATION
// -----------------------------------------------------------------------------
try {
    $pdo = get_db_connection();

    // Prepare a statement to find the user by their email address.
    // Using prepared statements is crucial to prevent SQL injection.
    $stmt = $pdo->prepare("SELECT id, full_name, email, password, email_verified FROM users WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch();

    // Check if a user was found AND if the provided password matches the hashed password in the database.
    if ($user && password_verify($password, $user['password'])) {

        // --- SUCCESSFUL LOGIN ---

        // Regenerate session ID to prevent session fixation attacks.
        session_regenerate_id(true);

        // Store user information in the session.
        $_SESSION['user_id'] = $user['id'];
        $_SESSION['user_name'] = $user['full_name'];
        $_SESSION['user_email'] = $user['email'];
        $_SESSION['is_logged_in'] = true;

        // Note: For a production system, you might want to check if $user['email_verified'] is true.
        // if (!$user['email_verified']) {
        //     $_SESSION['error_message'] = "Please verify your email address before logging in.";
        //     header('Location: ../index.php?page=login');
        //     exit();
        // }

        // Redirect to the user dashboard.
        header('Location: ../index.php?page=dashboard');
        exit();

    } else {
        // --- FAILED LOGIN ---
        // If user not found or password incorrect, show a generic error message
        // to avoid revealing which one was wrong (user enumeration).
        $_SESSION['error_message'] = "Invalid email or password.";
        header('Location: ../index.php?page=login');
        exit();
    }

} catch (PDOException $e) {
    // For a production environment, you should log this error instead of displaying it.
    // error_log("Login failed: " . $e->getMessage());
    $_SESSION['error_message'] = "An unexpected error occurred. Please try again later.";
    header('Location: ../index.php?page=login');
    exit();
}
