<?php
/**
 * User Management Handler
 *
 * This script handles all Create, Read, Update, Delete (CRUD) operations
 * for user management, initiated from the admin panel.
 */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once '../config.php';
require_once '../app/helpers.php';

// 1. SECURITY CHECKS
// -----------------------------------------------------------------------------
// Ensure only logged-in admins can access this.
if (!is_admin()) {
    $_SESSION['error_message'] = "Unauthorized access.";
    header('Location: ../admin/index.php?page=login');
    exit();
}

// Ensure the request is a POST request to prevent direct access.
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header('Location: ../admin/index.php?page=users');
    exit();
}

// 2. DETERMINE THE ACTION
// -----------------------------------------------------------------------------
$action = $_POST['action'] ?? '';

try {
    $pdo = get_db_connection();

    switch ($action) {
        // --- CASE: ADD A NEW USER ---
        case 'add_user':
            $full_name = trim($_POST['full_name'] ?? '');
            $email = trim($_POST['email'] ?? '');
            $password = $_POST['password'] ?? '';

            // Basic validation
            if (empty($full_name) || empty($email) || empty($password) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $_SESSION['error_message'] = "Invalid or incomplete data provided for new user.";
                header('Location: ../admin/index.php?page=users');
                exit();
            }

            // Check if email already exists
            $stmt_check = $pdo->prepare("SELECT id FROM users WHERE email = ?");
            $stmt_check->execute([$email]);
            if ($stmt_check->fetch()) {
                $_SESSION['error_message'] = "A user with that email address already exists.";
                header('Location: ../admin/index.php?page=users');
                exit();
            }

            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
            $stmt_insert = $pdo->prepare("INSERT INTO users (full_name, email, password, email_verified) VALUES (?, ?, ?, 1)"); // Admin-added users are pre-verified
            $stmt_insert->execute([$full_name, $email, $hashed_password]);

            $_SESSION['success_message'] = "User '{$full_name}' was successfully created.";
            break;

        // --- CASE: EDIT AN EXISTING USER ---
        case 'edit_user':
            $user_id = $_POST['user_id'] ?? 0;
            $full_name = trim($_POST['full_name'] ?? '');
            $email = trim($_POST['email'] ?? '');
            $password = $_POST['password'] ?? ''; // Optional: only update if provided

            if (empty($user_id) || empty($full_name) || empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
                $_SESSION['error_message'] = "Invalid or incomplete data for editing user.";
                header('Location: ../admin/index.php?page=users');
                exit();
            }

            if (!empty($password)) {
                // If a new password is set, update it.
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                $stmt_update = $pdo->prepare("UPDATE users SET full_name = ?, email = ?, password = ? WHERE id = ?");
                $stmt_update->execute([$full_name, $email, $hashed_password, $user_id]);
            } else {
                // If no new password, update only name and email.
                $stmt_update = $pdo->prepare("UPDATE users SET full_name = ?, email = ? WHERE id = ?");
                $stmt_update->execute([$full_name, $email, $user_id]);
            }

            $_SESSION['success_message'] = "User details have been successfully updated.";
            break;

        // --- CASE: DELETE A USER ---
        case 'delete_user':
            $user_id = $_POST['user_id'] ?? 0;

            if (empty($user_id)) {
                $_SESSION['error_message'] = "Invalid user ID for deletion.";
                header('Location: ../admin/index.php?page=users');
                exit();
            }

            // You might want to prevent an admin from deleting their own account via this interface
            // or add other business logic here.

            $stmt_delete = $pdo->prepare("DELETE FROM users WHERE id = ?");
            $stmt_delete->execute([$user_id]);

            $_SESSION['success_message'] = "User has been successfully deleted.";
            break;
        
        // --- DEFAULT CASE FOR INVALID ACTION ---
        default:
            $_SESSION['error_message'] = "Invalid action specified.";
            break;
    }

} catch (PDOException $e) {
    $_SESSION['error_message'] = "A database error occurred: " . $e->getMessage();
}

// 3. REDIRECT BACK TO THE USER MANAGEMENT PAGE
// -----------------------------------------------------------------------------
header('Location: ../admin/index.php?page=users');
exit();

