<?php
/**
 * Admin Edit User Page
 *
 * This page provides a form for an administrator to edit an existing user's account details.
 */

// The admin router (admin/index.php) handles the session and security checks.
include THEME_PATH . 'admin/header.php'; // Include the admin header

// 1. Get User ID from URL and Fetch User Data from Database
// -----------------------------------------------------------------------------
$user_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
$user = null;
$error_message = '';

if ($user_id) {
    try {
        $pdo = get_db_connection();
        $stmt = $pdo->prepare("SELECT id, full_name, email FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch();

        if (!$user) {
            $error_message = "User not found. This user may have been deleted.";
        }
    } catch (PDOException $e) {
        $error_message = "A database error occurred: " . $e->getMessage();
    }
} else {
    $error_message = "Invalid or missing User ID provided in the URL.";
}

?>

<div class="admin-content">
    <div class="container">
        <h1 class="page-title">Edit User</h1>
        
        <?php if ($error_message): ?>
            <!-- Display error if user ID is invalid or user not found -->
            <p class="page-subtitle">Could not load user data.</p>
            <div class="alert alert-danger"><?php echo e($error_message); ?></div>
            <a href="index.php?page=users" class="btn btn-secondary"><i class="fas fa-arrow-left"></i> Back to User List</a>
        <?php else: ?>
            <!-- Display the form if user is found -->
            <p class="page-subtitle">Update the details for user: <strong><?php echo e($user['full_name']); ?></strong></p>

            <div class="form-container">
                <form action="../actions/user_management_handler.php" method="POST">
                    <!-- Hidden inputs to specify the action and the user ID -->
                    <input type="hidden" name="action" value="edit_user">
                    <input type="hidden" name="user_id" value="<?php echo e($user['id']); ?>">

                    <div class="form-group">
                        <label for="full_name">Full Name</label>
                        <input type="text" id="full_name" name="full_name" class="form-control" value="<?php echo e($user['full_name']); ?>" required>
                    </div>

                    <div class="form-group">
                        <label for="email">Email Address</label>
                        <input type="email" id="email" name="email" class="form-control" value="<?php echo e($user['email']); ?>" required>
                    </div>

                    <div class="form-group">
                        <label for="password">New Password</label>
                        <input type="password" id="password" name="password" class="form-control" placeholder="Leave blank to keep current password">
                        <small class="form-text">If you enter a new password, it must be at least 8 characters long.</small>
                    </div>

                    <div class="form-actions">
                        <button type="submit" class="btn btn-primary"><i class="fas fa-save"></i> Update User</button>
                        <a href="index.php?page=users" class="btn btn-secondary">Cancel</a>
                    </div>
                </form>
            </div>
        <?php endif; ?>

    </div>
</div>

<?php
// Include the admin footer template.
include THEME_PATH . 'admin/footer.php';
?>
