<?php
/**
 * Google OAuth 2.0 Callback Handler (Multi-purpose)
 *
 * This script is the single redirect URI for all Google OAuth flows.
 * It handles the response from Google for both:
 * 1. Admin AI Learning authentication.
 * 2. User Single Sign-On (SSO) login/registration.
 */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config.php';
require_once 'vendor/autoload.php';
require_once 'app/helpers.php';

// Determine what action to perform based on a session variable
// set right before redirecting to Google.
$action = $_SESSION['oauth_action'] ?? 'admin_auth'; // Default to admin auth for backward compatibility/safety

// Unset the action immediately to prevent it from being accidentally reused.
unset($_SESSION['oauth_action']);

// =============================================================================
// ACTION 1: HANDLE USER LOGIN / REGISTRATION VIA GOOGLE SSO
// =============================================================================
if ($action === 'user_login') {
    try {
        $client = new Google_Client();
        $client->setAuthConfig('credentials.json');
        $client->setRedirectUri(GOOGLE_REDIRECT_URI);
        $client->setAccessType('offline');
        $client->addScope("email");
        $client->addScope("profile");

        // Exchange the authorization code for an access token.
        $token = $client->fetchAccessTokenWithAuthCode($_GET['code']);
        $client->setAccessToken($token);

        // Get user profile information from Google.
        $google_oauth = new Google_Service_Oauth2($client);
        $google_account_info = $google_oauth->userinfo->get();
        
        $google_id = $google_account_info->getId();
        $email = $google_account_info->getEmail();
        $full_name = $google_account_info->getName();

        $pdo = get_db_connection();

        // Check if user already exists in the database by their Google ID or email.
        $stmt = $pdo->prepare("SELECT * FROM users WHERE google_id = ? OR email = ?");
        $stmt->execute([$google_id, $email]);
        $user = $stmt->fetch();

        if ($user) {
            // User exists, log them in.
            $_SESSION['user_id'] = $user['id'];
            // If they registered via email first and are now using Google, link their Google ID.
            if (empty($user['google_id'])) {
                $update_stmt = $pdo->prepare("UPDATE users SET google_id = ? WHERE id = ?");
                $update_stmt->execute([$google_id, $user['id']]);
            }
        } else {
            // User does not exist, create a new account for them.
            $stmt_insert = $pdo->prepare("INSERT INTO users (google_id, email, full_name, email_verified, current_step) VALUES (?, ?, ?, 1, 1)");
            $stmt_insert->execute([$google_id, $email, $full_name]);
            $_SESSION['user_id'] = $pdo->lastInsertId();
        }

        // Set common session variables and redirect to the user dashboard.
        $_SESSION['user_name'] = $full_name;
        $_SESSION['is_logged_in'] = true;
        header('Location: index.php?page=dashboard');
        exit();

    } catch (Exception $e) {
        $_SESSION['error_message'] = 'Google Sign-In failed: ' . $e->getMessage();
        header('Location: index.php?page=login');
        exit();
    }
}

// =============================================================================
// ACTION 2: HANDLE ADMIN AUTHENTICATION FOR AI LEARNING (DEFAULT)
// =============================================================================
try {
    $client = new Google_Client();
    $client->setAuthConfig('credentials.json');
    $client->setRedirectUri(GOOGLE_REDIRECT_URI);
    $client->setAccessType('offline');
    $client->addScope(Google_Service_Drive::DRIVE_READONLY);

    $token = $client->fetchAccessTokenWithAuthCode($_GET['code']);
    $client->setAccessToken($token);

    if (array_key_exists('error', $token)) {
        throw new Exception(join(', ', $token));
    }

    // Google only sends a refresh token on the very first consent.
    // If we don't get one, we try to preserve the old one.
    if (!$client->getRefreshToken()) {
        $tokenPath = 'token.json';
        if (file_exists($tokenPath)) {
            $oldTokenData = json_decode(file_get_contents($tokenPath), true);
            if (isset($oldTokenData['refresh_token'])) {
                $token['refresh_token'] = $oldTokenData['refresh_token'];
            }
        }
    }
    
    // Save the complete token to a file.
    $tokenPath = 'token.json';
    if (!file_put_contents($tokenPath, json_encode($token))) {
         throw new Exception('Failed to save the access token to token.json. Check file permissions.');
    }

    $_SESSION['success_message'] = 'Google Drive has been successfully authenticated. You can now update the AI learnings.';
    header('Location: admin/index.php?page=learning');
    exit();

} catch (Exception $e) {
    $_SESSION['error_message'] = 'An error occurred during admin authentication: ' . $e->getMessage();
    header('Location: admin/index.php?page=learning');
    exit();
}

