<?php
/**
 * Main Front-End Controller
 *
 * This is the single entry point for all user-facing pages (non-admin).
 * It handles routing, security, and language selection before loading the
 * appropriate theme file to display to the user.
 */

// 1. BOOTSTRAPPING
// -----------------------------------------------------------------------------
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config.php';
require_once 'vendor/autoload.php';
require_once 'app/helpers.php';


// 2. LANGUAGE HANDLING
// -----------------------------------------------------------------------------
// Check if the user has submitted the language selection form.
// Uses the 'SUPPORTED_LANGUAGES' constant from config.php.
if (isset($_POST['lang']) && array_key_exists($_POST['lang'], SUPPORTED_LANGUAGES)) {
    $_SESSION['lang'] = $_POST['lang'];

    // FIXED: Ensure session data is written before redirecting to prevent race conditions.
    session_write_close();
    
    // Redirect to the same page to prevent form resubmission and preserve the clean URL.
    header('Location: ' . $_SERVER['REQUEST_URI']);
    exit();
}


// 3. ROUTING LOGIC
// -----------------------------------------------------------------------------
// Determine the requested page from the URL. The .htaccess file rewrites
// clean URLs (e.g., /login) to index.php?page=login.
// If no page is specified, default to 'dashboard' if logged in, or 'login' if not.
$page = $_GET['page'] ?? (is_logged_in() ? 'dashboard' : 'login');

// Whitelist of all valid, non-admin pages. This is a security measure
// to prevent loading arbitrary files.
$allowed_pages = [
    'login',
    'register',
    'forgot_password',
    'dashboard',
    'upload',
    'payment',
    'psy_test',
    'interview',
    'results'
];

if (in_array($page, $allowed_pages)) {

    // 4. SECURITY CHECKS
    // -----------------------------------------------------------------------------
    $is_auth_page = in_array($page, ['login', 'register', 'forgot_password']);

    // If a user is NOT logged in and tries to access a protected page (like dashboard)...
    if (!is_logged_in() && !$is_auth_page) {
        // ...redirect them to the login page using a clean URL.
        header('Location: login');
        exit();
    }

    // If a user IS logged in and tries to access an authentication page (like login)...
    if (is_logged_in() && $is_auth_page) {
        // ...redirect them to their dashboard.
        header('Location: dashboard');
        exit();
    }

    // 5. RENDER THE PAGE
    // -----------------------------------------------------------------------------
    // Determine the correct path to the theme file based on the page type.
    if ($is_auth_page) {
        $page_path = THEME_PATH . 'auth/' . $page . '.php';
    } else {
        $page_path = THEME_PATH . 'user/' . $page . '.php';
    }

    // Final check to ensure the theme file actually exists before including it.
    if (file_exists($page_path)) {
        include $page_path;
    } else {
        // This indicates a developer error (page in whitelist but file is missing).
        http_response_code(500);
        die("FATAL ERROR: The view file for '{$page}' is missing at: {$page_path}");
    }

} else {
    // If the requested page is not in our whitelist, it's a 404 Not Found error.
    http_response_code(404);
    // You could include a custom 404 page here for a better user experience.
    die("Error 404: Page not found.");
}

