<?php
/**
 * User Page: View Results (Step 5)
 *
 * Ini adalah halaman hasil akhir.
 *
 * MODIFIKASI:
 * 1. Cek jika laporan sudah ada di `interview_reports`.
 * 2. Jika belum, buat laporan baru.
 * 3. Saat membuat, ambil data psikotes dari tabel BARU (`user_psychotest_reports`)
 * atau tabel LAMA (`user_test_results`) sebagai fallback.
 * 4. Render HTML yang sesuai untuk tipe tes (DISC, PAPI, atau Simple).
 * 5. Simpan HTML laporan ke database.
 * 6. Tampilkan laporan (yang ada atau yang baru dibuat).
 */

// 1. Inisialisasi & Keamanan
// (Kita asumsikan main_header.php atau index.php sudah menjalankan session_start())

if (!is_logged_in()) {
    header('Location: index.php?page=login');
    exit();
}

// Pastikan pengguna berada di langkah yang benar
// (interview_handler.php seharusnya mengatur current_step = 5)
if ($_SESSION['current_step'] < 5) {
    // Jika mereka mencoba melompat, kembalikan ke dashboard
    header('Location: index.php?page=dashboard');
    exit();
}

$user_id = $_SESSION['user_id'];
$report_html = null;

try {
    $pdo = get_db_connection();

    // 1. PERIKSA APAKAH LAPORAN SUDAH PERNAH DIBUAT?
    // -------------------------------------------------------------------------
    $stmt_find = $pdo->prepare(
        "SELECT report_html_content FROM interview_reports WHERE user_id = ? ORDER BY id DESC LIMIT 1"
    );
    $stmt_find->execute([$user_id]);
    $existing_report = $stmt_find->fetchColumn();

    if ($existing_report) {
        // --- LAPORAN SUDAH ADA, LANGSUNG TAMPILKAN ---
        $report_html = $existing_report;
        
    } else {
        // --- LAPORAN BELUM ADA, BUAT LAPORAN BARU ---
        
        // 2. KUMPULKAN SEMUA DATA
        // ---------------------------------------------------------------------

        // 2a. Ambil Data Upload (CV & Job Ad) 
        $stmt_uploads = $pdo->prepare("SELECT file_type, original_filename FROM user_uploads WHERE user_id = ?");
        $stmt_uploads->execute([$user_id]);
        $uploads_raw = $stmt_uploads->fetchAll(PDO::FETCH_KEY_PAIR);
        $cv_filename = $uploads_raw['cv'] ?? 'N/A';
        $job_ad_filename = $uploads_raw['job_ad'] ?? 'N/A';

        // 2b. Ambil Data Wawancara 
        $stmt_interview = $pdo->prepare("SELECT * FROM interview_results WHERE user_id = ? ORDER BY id ASC");
        $stmt_interview->execute([$user_id]);
        $interview_results = $stmt_interview->fetchAll(PDO::FETCH_ASSOC);

        // 2c. Ambil Data Psikotes (LOGIKA BARU YANG CERDAS)
        $psy_html_block = ''; // Blok HTML yang akan kita isi

        // Coba ambil dari tabel KOMPLEKS (DISC/PAPI) terlebih dahulu
        $stmt_psy_complex = $pdo->prepare(
            "SELECT t.test_name, t.test_type, r.report_data 
             FROM user_psychotest_reports r
             JOIN psychological_tests t ON r.test_id = t.id
             WHERE r.user_id = ? ORDER BY r.id DESC LIMIT 1"
        );
        $stmt_psy_complex->execute([$user_id]);
        $complex_report = $stmt_psy_complex->fetch(PDO::FETCH_ASSOC);

        if ($complex_report) {
            // --- DITEMUKAN HASIL TES KOMPLEKS (DISC ATAU PAPI) ---
            $report_data = json_decode($complex_report['report_data'], true);
            $test_name = htmlspecialchars($complex_report['test_name']);
            $psy_html_block = "<h4><i class='fas fa-brain'></i> Profil Psikologis: {$test_name}</h4>";

            if ($complex_report['test_type'] === 'disc') {
                $scores = $report_data['scores']['graph3_change'] ?? [];
                $profile = htmlspecialchars($report_data['dominant_profile'] ?? 'N/A');
                $psy_html_block .= "<h5>Profil DISC (Graph 3: Change/Perceived)</h5>";
                $psy_html_block .= "<ul class='list-group list-group-flush'>";
                $psy_html_block .= "<li class='list-group-item d-flex justify-content-between align-items-center'><strong>Dominance (D):</strong> <span class='badge badge-primary badge-pill'>" . ($scores['D'] ?? 0) . "</span></li>";
                $psy_html_block .= "<li class='list-group-item d-flex justify-content-between align-items-center'><strong>Influence (I):</strong> <span class='badge badge-primary badge-pill'>" . ($scores['I'] ?? 0) . "</span></li>";
                $psy_html_block .= "<li class='list-group-item d-flex justify-content-between align-items-center'><strong>Steadiness (S):</strong> <span class='badge badge-primary badge-pill'>" . ($scores['S'] ?? 0) . "</span></li>";
                $psy_html_block .= "<li class='list-group-item d-flex justify-content-between align-items-center'><strong>Conscientiousness (C):</strong> <span class='badge badge-primary badge-pill'>" . ($scores['C'] ?? 0) . "</span></li>";
                $psy_html_block .= "</ul>";
                $psy_html_block .= "<p class='mt-3'><strong>Profil Dominan:</strong> {$profile}</p>";

            } elseif ($complex_report['test_type'] === 'papi') {
                $scores = $report_data['raw_scores'] ?? [];
                $psy_html_block .= "<h5>Profil PAPI Kostick (Skor Mentah)</h5>";
                $psy_html_block .= "<div class='row'>";
                foreach ($scores as $facet => $score) {
                    $psy_html_block .= "<div class='col-md-3 col-6 mb-2'><strong>" . htmlspecialchars($facet) . ":</strong> " . (int)$score . "</div>";
                }
                $psy_html_block .= "</div>";
            }
        
        } else {
            // --- TIDAK DITEMUKAN DI TABEL KOMPLEKS, CARI DI TABEL SIMPLE (LAMA) ---
            $stmt_psy_simple = $pdo->prepare(
                "SELECT t.test_name, r.total_score 
                 FROM user_test_results r
                 JOIN psychological_tests t ON r.test_id = t.id
                 WHERE r.user_id = ? AND t.test_type = 'simple' 
                 ORDER BY r.id DESC LIMIT 1"
            );
            $stmt_psy_simple->execute([$user_id]);
            $simple_report = $stmt_psy_simple->fetch(PDO::FETCH_ASSOC);

            if ($simple_report) {
                // Ditemukan hasil tes 'simple'
                $test_name = htmlspecialchars($simple_report['test_name']);
                $psy_html_block = "<h4><i class='fas fa-brain'></i> Profil Psikologis: {$test_name}</h4>";
                $psy_html_block .= "<h3>Skor Total: " . (int)$simple_report['total_score'] . "</h3>";
            } else {
                // Tidak ada data psikotes sama sekali
                $psy_html_block = "<h4><i class='fas fa-brain'></i> Profil Psikologis</h4><p>Tidak ditemukan data hasil tes psikologi.</p>";
            }
        }

        // 3. BUAT BLOK HTML LAPORAN
        // ---------------------------------------------------------------------
        // Kita gunakan output buffering untuk 'menangkap' HTML
        ob_start(); 
        ?>
        
        <style>
            .report-section { margin-bottom: 30px; padding: 20px; border: 1px solid #eee; border-radius: 8px; }
            .report-section h2 { border-bottom: 2px solid #f0f0f0; padding-bottom: 10px; }
            .qa-item { margin-bottom: 20px; }
            .qa-item p { margin-bottom: 5px; }
            .premium-lock { text-align: center; padding: 40px 20px; background: #fdfdfd; border: 2px dashed #ccc; border-radius: 8px; }
            .premium-lock .fas { font-size: 40px; color: #aaa; margin-bottom: 15px; }
        </style>
        
        <div class="report-header text-center mb-4">
            <h1>Laporan Analisis Kandidat</h1>
            <p class="lead">Laporan ini dibuat untuk <strong><?php echo htmlspecialchars($_SESSION['user_name']); ?></strong> pada <?php echo date('d M Y, H:i'); ?></p>
        </div>

        <div class="report-section" id="documents">
            <h2><i class="fas fa-file-alt"></i> Dokumen yang Dianalisis</h2>
            <ul class="list-group">
                <li class="list-group-item"><strong>CV Kandidat:</strong> <?php echo htmlspecialchars($cv_filename); ?></li>
                <li class="list-group-item"><strong>Lowongan Pekerjaan:</strong> <?php echo htmlspecialchars($job_ad_filename); ?></li>
            </ul>
        </div>

        <div class="report-section" id="psychology">
            <?php echo $psy_html_block; // Tampilkan blok HTML psikotes yang sudah kita buat ?>
        </div>

        <div class="report-section" id="interview">
            <h2><i class="fas fa-microphone"></i> Transkrip Wawancara AI</h2>
            <?php if (empty($interview_results)): ?>
                <p>Tidak ada data transkrip wawancara yang tersimpan.</p>
            <?php else: ?>
                <?php foreach ($interview_results as $item): ?>
                    <div class="qa-item">
                        <p><strong>Pertanyaan:</strong> <?php echo htmlspecialchars($item['question_text']); ?></p>
                        <p><em><strong>Jawaban:</strong> <?php echo nl2br(htmlspecialchars($item['answer_transcript'])); ?></em></p>
                    </div>
                    <hr>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>

        <div class="report-section" id="analysis">
            <h2><i class="fas fa-cogs"></i> Analisis & Rekomendasi (Premium)</h2>
            <div class="premium-lock">
                <i class="fas fa-lock"></i>
                <h4>Fitur Ini Terkunci</h4>
                <p>Analisis mendalam mengenai kecocokan antara CV, profil psikologis, dan jawaban wawancara Anda adalah fitur premium.</p>
                <p>Dapatkan wawasan perilaku, rekomendasi area pengembangan, dan skor kecocokan terperinci.</p>
                <a href="#" class="btn btn-success btn-lg mt-3" onclick="alert('Alur pembayaran premium belum diimplementasikan.');">
                    <i class="fas fa-unlock-alt"></i> Unlock Laporan Lengkap
                </a>
            </div>
        </div>
        
        <?php
        // Ambil HTML yang sudah di-buffer
        $report_html = ob_get_clean();

        // 4. SIMPAN LAPORAN HTML KE DATABASE
        // ---------------------------------------------------------------------
        $stmt_save = $pdo->prepare(
            "INSERT INTO interview_reports (user_id, report_html_content, generated_at) VALUES (?, ?, NOW())"
        );
        $stmt_save->execute([$user_id, $report_html]);
    }

} catch (Exception $e) {
    // Tangani error jika terjadi kegagalan
    $report_html = "<div class='alert alert-danger'>Terjadi error saat membuat laporan: " . $e->getMessage() . "</div>";
}
?>

<div class="container mt-5 mb-5">
    <div class="row justify-content-center">
        <div class="col-lg-10">
            
            <div class="card shadow-sm">
                <div class="card-header bg-light d-flex justify-content-between align-items-center">
                    <h3><i class="fas fa-poll"></i> Laporan Anda Telah Selesai</h3>
                    <button class="btn btn-secondary btn-sm" onclick="window.print();">
                        <i class="fas fa-print"></i> Cetak Laporan
                    </button>
                </div>
                <div class="card-body p-4 p-md-5" id="report-content">

                    <?php echo $report_html; ?>

                </div> </div> </div> </div> </div> ```