<?php
/**
 * Admin Psychological Test Management Page
 *
 * This page allows administrators to create, view, update, and delete
 * psychological test modules used in the system.
 */

// The admin router (admin/index.php) handles the session and security checks.
include THEME_PATH . 'admin/header.php'; // Include the admin header

// --- DATA FETCHING FOR TESTS ---
$tests = [];
try {
    $pdo = get_db_connection();

    // Fetch all psychological tests from the database.
    $stmt = $pdo->query("
        SELECT id, test_name, description, question_count, created_at
        FROM psychological_tests
        ORDER BY created_at DESC
    ");
    $tests = $stmt->fetchAll();

} catch (PDOException $e) {
    $db_error = "Could not fetch psychological tests. Error: " . $e->getMessage();
}

?>

<div class="admin-content">
    <div class="container">
        <div class="page-header">
            <h1 class="page-title">Test Management</h1>
            <!-- This link now correctly points to the add_test page -->
            <a href="index.php?page=add_test" class="btn btn-success"><i class="fas fa-plus"></i> Add New Test</a>
        </div>
        <p class="page-subtitle">Create, edit, and manage the psychological test modules for candidates.</p>

        <!-- Session Message Display (for add/edit/delete actions) -->
        <?php if (isset($_SESSION['success_message'])): ?>
            <div class="alert alert-success"><?php echo e($_SESSION['success_message']); unset($_SESSION['success_message']); ?></div>
        <?php endif; ?>
        <?php if (isset($_SESSION['error_message'])): ?>
            <div class="alert alert-danger"><?php echo e($_SESSION['error_message']); unset($_SESSION['error_message']); ?></div>
        <?php endif; ?>
        <?php if (isset($db_error)): ?>
            <div class="alert alert-danger"><?php echo e($db_error); ?></div>
        <?php endif; ?>


        <!-- Tests Table -->
        <div class="data-table-container">
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Test Name</th>
                        <th>Description</th>
                        <th>Questions</th>
                        <th>Date Created</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($tests)): ?>
                        <tr>
                            <td colspan="5" class="text-center">No psychological tests found. Click "Add New Test" to get started.</td>
                        </tr>
                    <?php else: ?>
                        <?php foreach ($tests as $test): ?>
                            <tr>
                                <td data-label="Test Name"><strong><?php echo e($test['test_name']); ?></strong></td>
                                <td data-label="Description"><?php echo e($test['description']); ?></td>
                                <td data-label="Questions">
                                    <a href="index.php?page=manage_questions&test_id=<?php echo e($test['id']); ?>"><?php echo e($test['question_count']); ?></a>
                                </td>
                                <td data-label="Date Created">
                                    <?php
                                        $date = new DateTime($test['created_at']);
                                        echo $date->format('d M Y');
                                    ?>
                                </td>
                                <td data-label="Actions">
                                    <!-- This link now correctly points to the edit_test page -->
                                    <a href="index.php?page=edit_test&id=<?php echo e($test['id']); ?>" class="btn-action btn-edit"><i class="fas fa-edit"></i> Edit</a>
                                    
                                    <!-- Delete button within a form for security -->
                                    <form action="../actions/test_management_handler.php" method="POST" style="display: inline-block;">
                                        <input type="hidden" name="action" value="delete_test">
                                        <input type="hidden" name="test_id" value="<?php echo e($test['id']); ?>">
                                        <button type="submit" class="btn-action btn-delete" onclick="return confirm('Are you sure you want to delete this test and all its questions?');">
                                            <i class="fas fa-trash-alt"></i> Delete
                                        </button>
                                    </form>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>

    </div>
</div>

<?php
// Include the admin footer template.
include THEME_PATH . 'admin/footer.php';
?>

