/**
 * Main JavaScript File for AI HR Profiler Theme
 *
 * This file contains scripts for interactive elements, primarily for the admin panel.
 *
 * Table of Contents:
 * 1. Mobile Sidebar Toggle
 */

// We wrap our code in a DOMContentLoaded event listener to ensure all HTML elements
// are loaded before we try to attach any events to them.
document.addEventListener('DOMContentLoaded', function() {

    /**
     * 1. Mobile Sidebar Toggle
     *
     * This functionality handles the opening and closing of the admin sidebar
     * on screen sizes where it is hidden by default (as defined in style.css).
     */

    // Find the button that is meant to toggle the sidebar.
    // This element would need to be added to the admin header.
    const sidebarToggler = document.getElementById('sidebar-toggle');

    // Find the sidebar element itself.
    const sidebar = document.querySelector('.admin-sidebar');
    
    // Find the main content area.
    const mainContent = document.querySelector('.admin-main-content');

    // Only proceed if all necessary elements are found on the page.
    // This prevents errors on pages that don't have a sidebar, like the user dashboard.
    if (sidebarToggler && sidebar && mainContent) {

        sidebarToggler.addEventListener('click', function() {
            // Toggle a class on the sidebar to trigger the show/hide transition.
            sidebar.classList.toggle('sidebar-visible');

            // Optionally, add an overlay to the main content when the sidebar is open.
            // This is a common UX pattern on mobile.
            if (sidebar.classList.contains('sidebar-visible')) {
                // Add an overlay element if it doesn't exist
                if (!document.querySelector('.sidebar-overlay')) {
                    const overlay = document.createElement('div');
                    overlay.className = 'sidebar-overlay';
                    
                    // Clicking the overlay should also close the sidebar
                    overlay.addEventListener('click', () => {
                        sidebar.classList.remove('sidebar-visible');
                        overlay.remove();
                    });

                    mainContent.appendChild(overlay);
                }
            } else {
                // Remove the overlay if it exists
                const overlay = document.querySelector('.sidebar-overlay');
                if (overlay) {
                    overlay.remove();
                }
            }
        });
    }

     /**
     * 2. AI Learning Progress Bar Handler
     *
     * This functionality shows a progress bar overlay when an admin
     * clicks one of the "Update Learnings" buttons.
     */
    
    // Find all forms used for learning (both CVs and Job Posters)
    const learningForms = document.querySelectorAll('.learning-form');
    
    // Find the progress overlay element
    const progressOverlay = document.getElementById('progress-overlay');

    // Attach an event listener to each form
    learningForms.forEach(form => {
        form.addEventListener('submit', function() {
            // Find the button within the submitted form
            const submitButton = form.querySelector('.btn-learn');

            // Disable the button to prevent multiple clicks
            if (submitButton) {
                submitButton.disabled = true;
                submitButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Processing...';
            }

            // Show the progress overlay if it exists
            if (progressOverlay) {
                progressOverlay.style.display = 'flex';
            }
        });
    });
});