<?php
/**
 * User AI Interview Page (with Timer, Transitions, Multi-language, and Error Handling)
 */
include THEME_PATH . 'header.php';

// --- LANGUAGE AND QUESTIONS SETUP ---
$current_lang = $_SESSION['lang'] ?? DEFAULT_LANG;

$speech_api_lang_map = [
    'id' => 'id-ID', 'en' => 'en-US', 'su' => 'su-ID', 'jv' => 'jv-ID',
];
$speech_lang_for_js = $speech_api_lang_map[$current_lang] ?? 'id-ID';

$interview_questions = [
    'id' => [
        "Ceritakan sebuah situasi spesifik di masa lalu di mana Anda dihadapkan pada sebuah proyek dengan tenggat waktu yang sangat ketat.",
        "Apa tugas atau tanggung jawab utama Anda dalam proyek tersebut?",
        "Jelaskan langkah-langkah atau tindakan konkret yang Anda ambil untuk memastikan proyek tersebut selesai tepat waktu.",
        "Apa hasil akhir dari tindakan Anda? Apakah proyek berhasil, dan apa yang Anda pelajari dari pengalaman tersebut?"
    ],
    'en' => [
        "Tell me about a specific situation in the past where you faced a project with a very tight deadline.",
        "What was your primary task or responsibility in that project?",
        "Describe the concrete steps or actions you took to ensure the project was completed on time.",
        "What was the final outcome of your actions? Was the project successful, and what did you learn from that experience?"
    ],
    'su' => [
        "Caritakeun kaayaan spésifik kapungkur dimana anjeun nyanghareupan proyék kalayan wates waktu anu ketat pisan.",
        "Naon pancén atanapi tanggung jawab utama anjeun dina proyék éta?",
        "Jelaskeun léngkah-léngkah atanapi tindakan kongkrit anu anjeun lakukeun pikeun mastikeun proyék réngsé dina waktosna.",
        "Naon hasil ahir tina tindakan anjeun? Naha proyék éta suksés, sareng naon anu anjeun pelajari tina pangalaman éta?"
    ],
    'jv' => [
        "Cariyosaken kahanan tartamtu ing jaman kepungkur nalika sampeyan ngadhepi proyék kanthi wates wektu sing ketat banget.",
        "Apa tugas utawa tanggung jawab utama sampeyan ing proyék kasebut?",
        "Jelasna langkah-langkah utawa tumindak konkrit sing sampeyan tindakake kanggo mesthekake proyék rampung ing wektu.",
        "Apa asil pungkasan saka tumindak sampeyan? Apa proyék kasebut kasil, lan apa sing sampeyan sinau saka pengalaman kasebut?"
    ]
];
$questions_for_js = $interview_questions[$current_lang] ?? $interview_questions['id'];

?>

<div class="user-dashboard">
    <div class="container">
        <div class="interview-container">
            </div>
        <div class="transition-overlay" id="transition-overlay" style="display: none;">
            <div class="transition-content">
                <h2 id="transition-text">Bersiaplah...</h2>
                <div class="countdown" id="transition-countdown">5</div>
            </div>
        </div>
    </div>
</div>

<style>
/* Revamped Styles for the AI Interview Interface */
.interview-container { max-width: 900px; margin: 40px auto; background: var(--card-bg); padding: 40px; border-radius: 12px; box-shadow: 0 8px 25px rgba(0,0,0,0.08); }
.interview-header { text-align: center; margin-bottom: 20px; }
.interview-main { display: flex; flex-direction: column; align-items: center; gap: 20px; }
.video-wrapper { position: relative; width: 100%; max-width: 640px; margin: 0 auto; border-radius: 12px; overflow: hidden; background: #000; box-shadow: 0 5px 15px rgba(0,0,0,0.1); aspect-ratio: 16 / 9; }
#avatar-video { width: 100%; height: 100%; object-fit: cover; }
#user-video { position: absolute; bottom: 15px; right: 15px; width: 150px; border-radius: 8px; border: 2px solid rgba(255,255,255,0.8); transform: scaleX(-1); }
.status-panel { width: 100%; text-align: center; }
.status-text { font-size: 20px; font-weight: 600; min-height: 50px; line-height: 1.5; }
.transcript-box { background: #f8f9fa; border: 1px solid var(--border-color); border-radius: 8px; padding: 15px; min-height: 60px; color: var(--text-secondary); font-style: italic; margin-top: 10px; }
.instruction-text { font-weight: 600; color: var(--primary-color); margin-top: 10px; }

/* Timer Styles */
.timer-container { position: relative; width: 100px; height: 100px; margin: 20px auto; }
.timer-progress-circle { transform: rotate(-90deg); }
.timer-progress-circle path { transition: all 1s linear; }
.timer-path-remaining { stroke-width: 8px; stroke: var(--primary-color); }
.timer-path-elapsed { stroke-width: 8px; stroke-linecap: round; stroke: #e9ecef; }
.timer-text { position: absolute; top: 50%; left: 50%; transform: translate(-50%, -50%); font-size: 24px; font-weight: 700; }
.timer-warning { color: #f57c00; } /* Orange for 20s */
.timer-danger { color: var(--danger-color); } /* Red for 10s */

/* Transition Overlay Styles */
.transition-overlay { position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.85); display: flex; justify-content: center; align-items: center; z-index: 10000; color: white; text-align: center; backdrop-filter: blur(5px); }
.transition-content h2 { font-size: 32px; }
.countdown { font-size: 72px; font-weight: 700; }

/* Video Error Message Style */
#video-error-container {
    display: none; /* Hidden by default */
    position: absolute;
    top: 0; left: 0; width: 100%; height: 100%;
    color: white;
    text-align: center;
    padding: 20px;
    flex-direction: column;
    justify-content: center;
    align-items: center;
    box-sizing: border-box;
}
#video-error-container i { font-size: 48px; margin-bottom: 20px; }
</style>

<script>
document.addEventListener('DOMContentLoaded', function () {
    const SPEECH_LANG = <?php echo json_encode($speech_lang_for_js); ?>;
    const questions = <?php echo json_encode($questions_for_js); ?>;
    const TIME_LIMIT = 180; // 3 minutes per question

    const interviewContainer = document.querySelector('.interview-container');
    const transitionOverlay = document.getElementById('transition-overlay');
    const transitionText = document.getElementById('transition-text');
    const transitionCountdown = document.getElementById('transition-countdown');
    
    let stream, currentQuestionIndex = 0, interviewData = [];
    let timerInterval = null, timeLeft = TIME_LIMIT;

    const synthesis = window.speechSynthesis;
    const recognition = new (window.SpeechRecognition || window.webkitSpeechRecognition)();
    recognition.lang = SPEECH_LANG;
    recognition.interimResults = true;
    recognition.continuous = true; // Listen continuously

    let avatarVideo, userVideo, statusText, transcriptBox, actionButton, timerTextElem;

    function initializeUI() {
        interviewContainer.innerHTML = `
            <div class="interview-header"><div class="step-icon active">4</div><h1><?php echo trans('interview_title'); ?></h1><p><?php echo trans('interview_subtitle'); ?></p></div>
            <div class="interview-main">
                <div class="video-wrapper">
                    <video id="avatar-video" playsinline autoplay muted loop>
                        <source src="https://storage.googleapis.com/gemini-prod/v1/1511252a-9e77-4933-b23a-8a56285494d6" type="video/mp4">
                    </video>
                    <div id="video-error-container">
                        <i class="fas fa-video-slash"></i>
                        <h3>Avatar Gagal Dimuat</h3>
                        <p>Ini bisa disebabkan oleh Adblocker, VPN, atau koneksi internet yang lambat. Coba nonaktifkan lalu segarkan halaman.</p>
                    </div>
                    <video id="user-video" playsinline autoplay muted></video>
                </div>
                <div class="status-panel">
                     <div id="timer-container" class="timer-container" style="display: none;">
                          <svg class="timer-progress-circle" viewBox="0 0 100 100"><g class="timer-path-elapsed"><path d="M 50, 50 m -45, 0 a 45,45 0 1,0 90,0 a 45,45 0 1,0 -90,0"></path></g><g class="timer-path-remaining"><path id="timer-path-remaining" d="M 50, 50 m -45, 0 a 45,45 0 1,0 90,0 a 45,45 0 1,0 -90,0" style="stroke-dasharray: 283;"></path></g></svg>
                          <span id="timer-text" class="timer-text"></span>
                     </div>
                    <p class="status-text" id="status-text"><?php echo trans('interview_wait_permission'); ?></p>
                    <div class="transcript-box" id="transcript-box"><i>...</i></div>
                    <p class="instruction-text" id="instruction-text" style="display:none;">Ucapkan <strong>"OK next"</strong> untuk melanjutkan ke pertanyaan berikutnya.</p>
                </div>
                <div class="interview-actions"><button id="action-button" class="btn btn-primary"><?php echo trans('interview_start_button'); ?></button></div>
            </div>`;
        
        avatarVideo = document.getElementById('avatar-video'); 
        userVideo = document.getElementById('user-video'); 
        statusText = document.getElementById('status-text'); 
        transcriptBox = document.getElementById('transcript-box');
        actionButton = document.getElementById('action-button');
        timerTextElem = document.getElementById('timer-text');
        
        avatarVideo.addEventListener('error', function() {
            console.error("Avatar video failed to load.");
            document.getElementById('video-error-container').style.display = 'flex';
            avatarVideo.style.display = 'none';
        });

        actionButton.addEventListener('click', startInterviewProcess);
    }

    function startTimer() {
        document.getElementById('timer-container').style.display = 'block';
        timeLeft = TIME_LIMIT;
        updateTimerUI();
        timerInterval = setInterval(() => {
            timeLeft--;
            updateTimerUI();
            if (timeLeft <= 0) {
                stopTimer();
                recognition.stop(); // This will trigger the onend event
            }
        }, 1000);
    }

    function stopTimer() { clearInterval(timerInterval); document.getElementById('timer-container').style.display = 'none'; }

    function updateTimerUI() {
        const minutes = Math.floor(timeLeft / 60); let seconds = timeLeft % 60; seconds = seconds < 10 ? '0' + seconds : seconds;
        timerTextElem.textContent = `${minutes}:${seconds}`;
        const circle = document.getElementById('timer-path-remaining');
        const fraction = timeLeft / TIME_LIMIT; const offset = fraction * 283;
        circle.style.strokeDasharray = `${offset} 283`;
        timerTextElem.classList.remove('timer-warning', 'timer-danger');
        circle.style.stroke = 'var(--primary-color)';
        if (timeLeft <= 10) { timerTextElem.classList.add('timer-danger'); circle.style.stroke = 'var(--danger-color)'; } 
        else if (timeLeft <= 20) { timerTextElem.classList.add('timer-warning'); circle.style.stroke = '#f57c00'; }
    }

    function showTransition(callback) {
        transitionText.textContent = "Pertanyaan Berikutnya...";
        transitionOverlay.style.display = 'flex';
        let countdown = 5; transitionCountdown.textContent = countdown;
        const countdownInterval = setInterval(() => {
            countdown--; transitionCountdown.textContent = countdown;
            if (countdown <= 0) { clearInterval(countdownInterval); transitionOverlay.style.display = 'none'; if(callback) callback(); }
        }, 1000);
    }

    async function startInterviewProcess() {
        try {
            stream = await navigator.mediaDevices.getUserMedia({ video: true, audio: true });
            userVideo.srcObject = stream;
            statusText.textContent = "Bersiaplah, wawancara akan dimulai...";
            actionButton.disabled = true;
            actionButton.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Memulai...';
            setTimeout(() => { actionButton.style.display = 'none'; showTransition(askQuestion); }, 2000);
        } catch (err) { statusText.textContent = "Error: Izin kamera/mikrofon ditolak."; console.error(err); }
    }
    
    function askQuestion() {
        if (currentQuestionIndex < questions.length) {
            const questionText = questions[currentQuestionIndex];
            statusText.textContent = questionText;
            document.getElementById('instruction-text').style.display = 'block';
            const utterance = new SpeechSynthesisUtterance(questionText);
            utterance.lang = SPEECH_LANG;
            utterance.onend = () => { 
                setTimeout(() => { 
                    startListening(); 
                    startTimer(); 
                }, 500); 
            };
            synthesis.speak(utterance);
        } else { finishInterview(); }
    }

    /**
     * --- THIS IS THE CORRECTED FUNCTION ---
     */
    function startListening() {
        transcriptBox.innerHTML = `<i><?php echo trans('interview_listening'); ?></i>`;
        recognition.start();

        let finalTranscript = '';

        recognition.onresult = (event) => {
            let interimTranscript = '';
            // Reset final transcript for this listening session to rebuild it
            finalTranscript = ''; 
            for (let i = 0; i < event.results.length; ++i) {
                if (event.results[i].isFinal) {
                    finalTranscript += event.results[i][0].transcript;
                } else {
                    interimTranscript += event.results[i][0].transcript;
                }
            }
            transcriptBox.textContent = finalTranscript + interimTranscript;

            // Check if the final, confirmed part of the speech contains the command
            if (finalTranscript.toLowerCase().includes('ok next') || finalTranscript.toLowerCase().includes('okay next')) {
                recognition.stop(); // Stop listening, which will trigger the 'onend' event
            }
        };
        
        recognition.onend = () => {
            // Get the full transcript from the box
            const fullTranscript = transcriptBox.textContent.trim();
            const advanceCommand = fullTranscript.toLowerCase().includes('ok next') || fullTranscript.toLowerCase().includes('okay next');

            // We advance to the next question ONLY IF the user gave the command OR the time ran out.
            if (advanceCommand || timeLeft <= 0) {
                stopTimer();
                const cleanedAnswer = fullTranscript.replace(/ok next|okay next/gi, '').trim();
                
                interviewData.push({ 
                    question: questions[currentQuestionIndex], 
                    answer: cleanedAnswer || '(no answer provided)' 
                });
                
                currentQuestionIndex++;
                showTransition(askQuestion); // Proceed to the next question
            } else {
                // If recognition stops for any other reason (e.g., brief silence),
                // and the interview isn't over, just restart it for the same question.
                if (currentQuestionIndex < questions.length) {
                    try {
                        recognition.start();
                    } catch (e) {
                        console.error("Recognition could not be restarted:", e);
                    }
                }
            }
        };

        recognition.onerror = (event) => {
            console.error('Speech recognition error:', event.error);
            // Optionally handle specific errors, like 'no-speech'
            if (event.error === 'no-speech') {
                 transcriptBox.innerHTML = `<i>Tidak terdeteksi suara. Mohon bicara lebih jelas.</i>`;
            }
        };
    }

    function finishInterview() {
        statusText.textContent = "Wawancara selesai. Menganalisis hasil Anda...";
        document.getElementById('instruction-text').style.display = 'none';
        if(stream) stream.getTracks().forEach(track => track.stop());

        // Make sure transcript box shows final message
        transcriptBox.innerHTML = `<i>Analisis sedang diproses...</i>`;

        fetch('actions/interview_handler.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({ interviewData: interviewData })
        })
        .then(response => response.json())
        .then(data => {
            if (data.status === 'success') { window.location.href = 'index.php?page=results'; } 
            else { statusText.textContent = `Error: ${data.message}`; }
        })
        .catch(err => { 
            console.error('Fetch Error:', err);
            statusText.textContent = 'Error: Gagal menyimpan hasil wawancara.'; 
        });
    }

    initializeUI();
});
</script>

<?php
include THEME_PATH . 'footer.php';
?>