<?php
/**
 * User Psychological Test Page
 *
 * This is Step 3 of the AI interview process. The user takes a psychological
 * test which is dynamically loaded from the database.
 */

// The main router (index.php) handles session and security checks.
include THEME_PATH . 'header.php'; // Includes the main header

// --- DATA FETCHING FOR THE TEST ---
$test = null;
$questions = [];
$error_message = '';

try {
    $pdo = get_db_connection();

    // Find the first test that has at least one question.
    $stmt_test = $pdo->query("
        SELECT t.id, t.test_name, t.description 
        FROM psychological_tests t
        JOIN test_questions q ON t.id = q.test_id
        GROUP BY t.id
        HAVING COUNT(q.id) > 0
        ORDER BY t.id ASC 
        LIMIT 1
    ");
    $test = $stmt_test->fetch();

    if (!$test) {
        $error_message = "No psychological tests with questions are currently available. Please go to the admin panel to add questions to a test.";
    } else {
        // Fetch all questions and their associated answers for this test
        $stmt_questions = $pdo->prepare("
            SELECT q.id AS question_id, q.question_text, a.id AS answer_id, a.answer_text
            FROM test_questions q
            LEFT JOIN test_answers a ON q.id = a.question_id
            WHERE q.test_id = ?
            ORDER BY q.id, a.id
        ");
        $stmt_questions->execute([$test['id']]);
        $results = $stmt_questions->fetchAll();

        // Group answers by question
        foreach ($results as $row) {
            if (!isset($questions[$row['question_id']])) {
                $questions[$row['question_id']] = [
                    'question_text' => $row['question_text'],
                    'answers' => []
                ];
            }
            if ($row['answer_id']) {
                $questions[$row['question_id']]['answers'][] = [
                    'answer_id' => $row['answer_id'],
                    'answer_text' => $row['answer_text']
                ];
            }
        }
    }
} catch (PDOException $e) {
    $error_message = "A database error occurred while loading the test: " . $e->getMessage();
}
?>

<div class="user-dashboard">
    <div class="container">
        <div class="test-container">
            
            <?php if ($error_message): ?>
                <div class="test-header"><h1>Test Not Ready</h1></div>
                <div class="alert alert-danger"><?php echo e($error_message); ?></div>
                <a href="index.php?page=dashboard" class="btn btn-secondary">Back to Dashboard</a>

            <?php elseif ($test && !empty($questions)): ?>
                <div class="test-header">
                    <div class="step-icon active">3</div>
                    <h1><?php echo e($test['test_name']); ?></h1>
                    <p><?php echo e($test['description']); ?></p>
                </div>
                
                <div class="test-progress">
                    <div class="progress-bar-test">
                        <div class="progress-bar-test-fill" id="test-progress-fill"></div>
                    </div>
                    <span id="test-progress-text">Question 1 of <?php echo count($questions); ?></span>
                </div>

                <form action="actions/psy_test_handler.php" method="POST" id="test-form">
                    <input type="hidden" name="test_id" value="<?php echo e($test['id']); ?>">
                    
                    <div class="slides-container">
                        <?php $question_number = 0; ?>
                        <?php foreach ($questions as $question_id => $q_data): ?>
                            <?php $question_number++; ?>
                            <div class="question-slide <?php echo ($question_number === 1) ? 'active' : ''; ?>" data-question-number="<?php echo $question_number; ?>">
                                <h2 class="question-text"><?php echo $question_number . '. ' . e($q_data['question_text']); ?></h2>
                                <div class="answers-wrapper">
                                    <?php if (empty($q_data['answers'])): ?>
                                        <p class="no-answers-warning">This question has no answers. Please contact an admin.</p>
                                    <?php else: ?>
                                        <?php foreach ($q_data['answers'] as $answer): ?>
                                            <label class="answer-option">
                                                <input type="radio" name="answers[<?php echo $question_id; ?>]" value="<?php echo e($answer['answer_id']); ?>" required>
                                                <span class="answer-text"><?php echo e($answer['answer_text']); ?></span>
                                            </label>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>

                    <div class="test-navigation">
                        <button type="button" id="next-btn" class="btn btn-primary">Next Question &rarr;</button>
                        <button type="submit" id="submit-btn" class="btn btn-success" style="display: none;">
                            <i class="fas fa-check-circle"></i> Complete Test & Proceed
                        </button>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    </div>
</div>

<style>
/* FIXED: Revamped styles for the psychological test page */
.test-container { max-width: 800px; margin: 40px auto; background: var(--card-bg); padding: 40px; border-radius: 12px; box-shadow: 0 8px 25px rgba(0,0,0,0.08); }
.test-header { text-align: center; margin-bottom: 30px; }
.test-header .step-icon { margin: 0 auto 15px auto; }
.test-header h1 { font-size: 28px; margin: 0 0 10px 0; }
.test-progress { margin-bottom: 40px; }
.progress-bar-test { height: 10px; background: #e9ecef; border-radius: 5px; overflow: hidden; }
.progress-bar-test-fill { height: 100%; width: 0%; background: var(--primary-color); transition: width 0.4s ease; }
#test-progress-text { display: block; text-align: center; margin-top: 10px; font-size: 14px; color: var(--text-secondary); }

.slides-container {
    position: relative;
    overflow: hidden;
    min-height: 250px;
}
.question-slide {
    display: none;
    animation: fadeIn 0.5s ease-in-out;
}
.question-slide.active {
    display: block;
}
@keyframes fadeIn {
    from { opacity: 0; transform: translateY(10px); }
    to { opacity: 1; transform: translateY(0); }
}

.question-text { font-size: 22px; margin-bottom: 30px; text-align: center; line-height: 1.5; }
.answers-wrapper { display: flex; flex-direction: column; gap: 15px; }
.answer-option { display: block; background: #f8f9fa; border: 2px solid var(--border-color); border-radius: 8px; padding: 15px 20px; cursor: pointer; transition: all 0.2s ease; }
.answer-option:hover { border-color: var(--primary-color); transform: translateY(-2px); }
.answer-option input[type="radio"] { display: none; }
.answer-option input[type="radio"]:checked + .answer-text { font-weight: 600; color: var(--primary-color); }
.answer-option input[type="radio"]:checked + .answer-text::before { content: '\f058'; font-family: 'Font Awesome 5 Free'; font-weight: 900; margin-right: 12px; color: var(--primary-color); }
.no-answers-warning { text-align: center; color: var(--danger-color); font-weight: 600; }
.test-navigation { text-align: center; margin-top: 40px; }
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('test-form');
    if (!form) return;

    const slides = form.querySelectorAll('.question-slide');
    const nextBtn = document.getElementById('next-btn');
    const submitBtn = document.getElementById('submit-btn');
    const progressFill = document.getElementById('test-progress-fill');
    const progressText = document.getElementById('test-progress-text');
    
    let currentSlide = 0;
    const totalSlides = slides.length;

    function updateProgress() {
        if(totalSlides === 0) return;
        const progressPercentage = ((currentSlide + 1) / totalSlides) * 100;
        progressFill.style.width = `${progressPercentage}%`;
        progressText.textContent = `Question ${currentSlide + 1} of ${totalSlides}`;
    }

    function showSlide(index) {
        slides.forEach((slide, i) => {
            slide.classList.toggle('active', i === index);
        });
        updateProgress();

        if (currentSlide === totalSlides - 1) {
            nextBtn.style.display = 'none';
            submitBtn.style.display = 'inline-block';
        } else {
            nextBtn.style.display = 'inline-block';
            submitBtn.style.display = 'none';
        }
    }

    nextBtn.addEventListener('click', function() {
        const currentInputs = slides[currentSlide].querySelectorAll('input[type="radio"]');
        const isAnswered = Array.from(currentInputs).some(input => input.checked);

        if (!isAnswered) {
            alert('Please select an answer before proceeding.');
            return;
        }

        if (currentSlide < totalSlides - 1) {
            currentSlide++;
            showSlide(currentSlide);
        }
    });

    if(totalSlides > 0) {
        showSlide(0);
    }
});
</script>

<?php
// Include the standard user footer.
include THEME_PATH . 'footer.php';
?>

