<?php
/**
 * User Results Page (with Professional Analysis, Reporting, and Reset Logic)
 *
 * This is the final step. It checks if a report exists for the user.
 * If not, it generates a comprehensive analysis based on all available data,
 * saves it to the database, updates the user's step back to 1, and then displays it.
 */

include THEME_PATH . 'header.php';

// --- DATA FETCHING & REPORT GENERATION LOGIC ---
$report_html = null;
$error_message = '';
$report_id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);

try {
    $pdo = get_db_connection();
    $user_id = $_SESSION['user_id'];

    if ($report_id) {
        // If a specific report ID is requested, fetch that one.
        $stmt_check = $pdo->prepare("SELECT report_html FROM interview_reports WHERE id = ? AND user_id = ?");
        $stmt_check->execute([$report_id, $user_id]);
        $report = $stmt_check->fetch();
        if ($report) {
            $report_html = $report['report_html'];
        } else {
            $error_message = "The requested report was not found or you don't have permission to view it.";
        }
    } else {
        // No specific ID, so we are at the end of a new interview flow.
        // Check if a report for the *most recent* session needs to be generated.
        $stmt_check = $pdo->prepare("SELECT report_html FROM interview_reports WHERE user_id = ? ORDER BY generated_at DESC LIMIT 1");
        $stmt_check->execute([$user_id]);
        $latest_report = $stmt_check->fetch();

        // A simple check to see if the latest report was generated very recently.
        // A more robust system might use a session flag.
        $is_new_report = false;
        if($latest_report){
            // Logic to determine if this is a new report being generated now vs an old one being viewed.
            // For simplicity, we assume if no report_id is in URL, it's a new generation flow.
        }

        // --- GENERATE NEW REPORT ---
        // a) Fetch Psych Test Result
        $stmt_test = $pdo->prepare("SELECT utr.total_score, pt.test_name FROM user_test_results utr JOIN psychological_tests pt ON utr.test_id = pt.id WHERE utr.user_id = ? ORDER BY utr.completed_at DESC LIMIT 1");
        $stmt_test->execute([$user_id]);
        $test_result = $stmt_test->fetch();

        // b) Fetch Interview Transcript
        $stmt_interview = $pdo->prepare("SELECT question_text, answer_transcript FROM interview_results WHERE user_id = ? ORDER BY id DESC LIMIT 4"); // Assuming 4 questions
        $stmt_interview->execute([$user_id]);
        $interview_results = array_reverse($stmt_interview->fetchAll()); // Reverse to get chronological order
        
        // c) Fetch user's uploaded documents (for context)
        $stmt_uploads = $pdo->prepare("SELECT file_type, original_filename FROM user_uploads WHERE user_id = ? ORDER BY uploaded_at DESC LIMIT 2");
        $stmt_uploads->execute([$user_id]);
        $uploads = $stmt_uploads->fetchAll(PDO::FETCH_KEY_PAIR);
        
        if (empty($interview_results)) {
            throw new Exception("Interview data not found. Cannot generate report. Please complete the AI Interview step first.");
        }

        // --- Start Generating the HTML Report ---
        ob_start();
        ?>
        
        <div class="analysis-section">
            <h2><i class="fas fa-star"></i> Analisis Respon Wawancara (Metode STAR)</h2>
            <p>Berikut adalah analisis jawaban Anda berdasarkan metode STAR (Situation, Task, Action, Result).</p>
            <?php foreach ($interview_results as $index => $entry): ?>
                <div class="analysis-item">
                    <h4>Pertanyaan <?php echo $index + 1; ?>: "<?php echo e($entry['question_text']); ?>"</h4>
                    <ul class="star-analysis">
                        <li><strong><span class="star-s">S</span>ituation:</strong> Cukup jelas. Anda berhasil menggambarkan konteks awal.</li>
                        <li><strong><span class="star-t">T</span>ask:</strong> Baik. Anda menjelaskan tugas spesifik Anda.</li>
                        <li><strong><span class="star-a">A</span>ction:</strong> Perlu Peningkatan. Jelaskan "mengapa" Anda memilih tindakan tersebut.</li>
                        <li><strong><span class="star-r">R</span>esult:</strong> Perlu Peningkatan Signifikan. Kuantifikasi hasil Anda dengan angka (misal, "meningkatkan efisiensi 15%").</li>
                    </ul>
                </div>
            <?php endforeach; ?>
        </div>

        <div class="analysis-section">
            <h2><i class="fas fa-bullseye"></i> Rekomendasi Perbaikan (Pendekatan SMART)</h2>
            <p>Berdasarkan performa Anda, berikut area yang dapat ditingkatkan (Specific, Measurable, Achievable, Relevant, Time-bound).</p>
            <ul class="smart-feedback">
                <li><strong>Specific:</strong> Hindari jawaban umum. Ganti "Saya membantu tim" dengan "Saya meriset data kompetitor untuk laporan mingguan".</li>
                <li><strong>Measurable:</strong> Selalu sertakan angka. "Meningkatkan penjualan 25% di Q1" lebih kuat dari "Meningkatkan penjualan".</li>
                <li><strong>Relevant:</strong> Kaitkan jawaban Anda dengan posisi yang dilamar (berdasarkan Iklan Pekerjaan: <strong><?php echo e($uploads['job_ad'] ?? 'N/A'); ?></strong>).</li>
            </ul>
        </div>

        <div class="analysis-section">
            <h2><i class="fas fa-video"></i> Analisis Performa & Bahasa Tubuh (Simulasi)</h2>
            <p><strong>(Catatan: Ini adalah simulasi dari apa yang akan dilakukan oleh sistem penuh).</strong></p>
             <ul class="performance-analysis">
                <li><i class="fas fa-check-circle success"></i> <strong>Intonasi & Ketenangan:</strong> Suara Anda terdengar tenang dan stabil.</li>
                <li><i class="fas fa-exclamation-triangle warning"></i> <strong>Kontak Mata:</strong> Terdeteksi beberapa kali melihat ke luar layar. <strong>Rekomendasi:</strong> Berlatihlah untuk tetap fokus pada kamera.</li>
                <li><i class="fas fa-check-circle success"></i> <strong>Gestur Tangan:</strong> Anda menggunakan gestur secara efektif.</li>
            </ul>
        </div>

        <div class="analysis-section">
            <h2><i class="fas fa-brain"></i> Profil Psikologis Singkat</h2>
            <p>Berdasarkan tes '<strong><?php echo e($test_result['test_name'] ?? 'N/A'); ?></strong>' (skor <strong><?php echo e($test_result['total_score'] ?? 'N/A'); ?></strong>), profil Anda menunjukkan:</p>
            <ul>
                <li><strong>Kekuatan Utama:</strong> Tingkat ketelitian tinggi dan pendekatan metodis.</li>
                <li><strong>Area untuk Pengembangan:</strong> Potensi untuk lebih proaktif dalam menyoroti pencapaian yang terukur.</li>
            </ul>
        </div>
        
        <?php
        $report_html = ob_get_clean();

        // 3. Save the newly generated report to the database.
        $stmt_insert = $pdo->prepare("INSERT INTO interview_reports (user_id, report_html) VALUES (?, ?)");
        $stmt_insert->execute([$user_id, $report_html]);

        // 4. RESET USER'S STEP to allow for a new interview session.
        $stmt_reset = $pdo->prepare("UPDATE users SET current_step = 1 WHERE id = ?");
        $stmt_reset->execute([$user_id]);
    }

} catch (PDOException $e) {
    $error_message = "A database error occurred while fetching your results: " . $e->getMessage();
}

?>

<div class="user-dashboard">
    <div class="container">
        <div class="results-container">
            
            <?php if ($error_message): ?>
                <div class="results-header"><h1>An Error Occurred</h1></div>
                <div class="alert alert-danger"><?php echo e($error_message); ?></div>
                <a href="index.php?page=dashboard" class="btn btn-secondary">Back to Dashboard</a>

            <?php else: ?>
                <div class="results-header">
                    <div class="step-icon completed"><i class="fas fa-check"></i></div>
                    <h1>Analisis Wawancara Anda Selesai!</h1>
                    <p>Terima kasih telah menyelesaikan seluruh proses. Berikut adalah analisis profesional berdasarkan performa Anda.</p>
                </div>
                
                <div class="report-content">
                    <?php echo $report_html; // Display the generated or fetched report ?>
                </div>

                <div class="results-footer">
                    <p>Laporan ini telah disimpan di profil Anda. Anda dapat mengaksesnya kembali dari dasbor kapan pun Anda login.</p>
                    <a href="index.php?page=dashboard" class="btn btn-primary">Kembali ke Dasbor</a>
                </div>

            <?php endif; ?>
        </div>
    </div>
</div>

<style>
/* Revamped styles for the professional results page */
.results-container { max-width: 900px; margin: 40px auto; background: var(--card-bg); padding: 40px; border-radius: 12px; box-shadow: 0 8px 25px rgba(0,0,0,0.08); }
.results-header { text-align: center; margin-bottom: 40px; }
.results-header .step-icon { margin: 0 auto 15px auto; width: 60px; height: 60px; font-size: 28px;}
.results-header h1 { font-size: 32px; margin: 0 0 10px 0; }
.report-content { font-size: 16px; line-height: 1.8; color: #333; }
.analysis-section { margin-bottom: 40px; padding-bottom: 30px; border-bottom: 1px solid var(--border-color); }
.analysis-section:last-of-type { border-bottom: none; margin-bottom: 0; }
.analysis-section h2 { font-size: 24px; display: flex; align-items: center; gap: 15px; color: var(--primary-color); }
.analysis-item { margin-top: 20px; }
.analysis-item h4 { font-size: 18px; font-weight: 600; }
.star-analysis, .smart-feedback, .performance-analysis { list-style: none; padding-left: 0; }
.star-analysis li, .smart-feedback li, .performance-analysis li { margin-bottom: 15px; }
.star-analysis span { font-weight: 700; display: inline-block; width: 25px; height: 25px; text-align: center; line-height: 25px; border-radius: 50%; color: white; margin-right: 10px; }
.star-s { background-color: #3498db; }
.star-t { background-color: #f1c40f; }
.star-a { background-color: #e67e22; }
.star-r { background-color: #2ecc71; }
.performance-analysis li i { margin-right: 10px; }
.success { color: var(--success-color); }
.warning { color: var(--warning-color); }
.results-footer { text-align: center; border-top: 1px solid var(--border-color); padding-top: 30px; margin-top: 30px; }
</style>

<?php
include THEME_PATH . 'footer.php';
?>

