<?php
/**
 * Email Verification Handler
 *
 * This script is accessed when a user clicks the verification link in their email.
 * It validates the token and activates the user's account.
 */

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once 'config.php';
require_once 'app/helpers.php';

// 1. Get the token from the URL
// -----------------------------------------------------------------------------
$token = $_GET['token'] ?? '';

if (empty($token)) {
    // If no token is provided, redirect with an error.
    $_SESSION['error_message'] = "Invalid verification link. The token is missing.";
    header('Location: index.php?page=login');
    exit();
}

// 2. Find the user with the matching token
// -----------------------------------------------------------------------------
try {
    $pdo = get_db_connection();

    // Find a user who has this token and whose email is not yet verified.
    $stmt = $pdo->prepare("SELECT id FROM users WHERE verification_token = ? AND email_verified = 0");
    $stmt->execute([$token]);
    $user = $stmt->fetch();

    if ($user) {
        // --- SUCCESS: User found, activate their account ---

        // Set email_verified to 1 (true) and clear the token so it can't be used again.
        $update_stmt = $pdo->prepare("UPDATE users SET email_verified = 1, verification_token = NULL WHERE id = ?");
        $update_stmt->execute([$user['id']]);

        // Set a success message and redirect to the login page.
        $_SESSION['success_message'] = "Your email has been successfully verified! You can now log in.";
        header('Location: index.php?page=login');
        exit();
        
    } else {
        // --- FAILURE: No user found or account already verified ---
        // Show a generic message to avoid revealing information.
        $_SESSION['error_message'] = "This verification link is invalid or has already been used.";
        header('Location: index.php?page=login');
        exit();
    }

} catch (PDOException $e) {
    // In production, log the error.
    error_log("Email verification failed: " . $e->getMessage());
    $_SESSION['error_message'] = "An unexpected database error occurred. Please try again later.";
    header('Location: index.php?page=login');
    exit();
}
